from zope.interface import implements, Interface
from zope.component import adapts, getMultiAdapter
from zope.formlib import form
from zope.event import notify

from AccessControl import getSecurityManager
from Acquisition import aq_inner

from Products.CMFDefault.formlib.schema import SchemaAdapterBase
from Products.CMFPlone.interfaces import IPloneSiteRoot

from Products.CMFCore.utils import getToolByName
from Products.statusmessages.interfaces import IStatusMessage

from plone.fieldsets.fieldsets import FormFieldsets
from plone.app.form.validators import null_validator

from plone.app.controlpanel.form import ControlPanelForm
from plone.app.controlpanel.events import ConfigurationChangedEvent

from plone.protect import CheckAuthenticator

from betahaus.openmember import MessageFactory as _
from betahaus.openmember import CATALOG_ID, OPENMEMBER_MANAGE_DATABASE
from betahaus.openmember.interfaces import IOMControlPanelDatabaseSchema, IOMControlPanelForm


_controlpanel_forms = []

def registerForm(interface):
    """Registers an interface as a sub controlpanel form in the OpenMember control panel.
    The interface needs to inherit from the betahaus.openmember.interfaces.IOMControlPanelForm 
    in order to be displayed as a sub control panel."""
    
    if interface not in _controlpanel_forms:
        _controlpanel_forms.append(interface)

class OMControlPanelAdapter(SchemaAdapterBase):
    """Control panel adapter for the standard database settings"""
    implements(IOMControlPanelDatabaseSchema)
    adapts(IPloneSiteRoot)
    
    def getId(self):
        return 'database'
    
    def getLabel(self):
        return  _('Database')
    
    def __init__(self, context):
        super(OMControlPanelAdapter, self).__init__(context)
        self.omcatalog = getToolByName(context, CATALOG_ID)
        self.pprop = getToolByName(context, 'portal_properties')
        self.default_charset = self.pprop.site_properties.default_charset


    def get_fields_to_index(self):
        result = []
        for index in self.omcatalog.mirrorableIndexes():
            item = self.omcatalog.Indexes.get(index)
            if item.id.startswith('sortable'):
                continue
            label = getattr(item, 'label', None)
            if not label:
                label = item.id
                
            result.append("%s|%s|%s" % (item.id, item.meta_type, label.decode(self.default_charset)) )

        return result

    def set_fields_to_index(self, value):

        # add indexes and columns that are not in the catalog yet.
        # prune the omindex, omcolumns so we know if there are any unneeded indexes, columns
        indexes = []
        for row in value:
            values = row.strip().split("|")
            if len(values) == 3:
                name = values[0]
                meta_type = values[1]
                label = values[2]
                
            elif len(values) == 2:
                name = values[0]
                meta_type = values[1]
                label = values[0]
            else:
                raise ValueError('wrong number of values, must be 2 or 3.')
            
            indexes.append((name.encode(self.default_charset),meta_type.encode(self.default_charset), label.encode(self.default_charset)))
                    
        self.omcatalog.setMirrorFields(indexes)
        self.omcatalog.clearFindAndRebuild()

    fields_to_index = property(get_fields_to_index,
                               set_fields_to_index)
    
    def get_title_fields(self):
        """The address fields are stored in the properties tool, on the openmember_properties sheet lines field called address_fields
        """
        portal_properties = getToolByName(self.context, 'portal_properties', None)
        openmember_properties = getattr(portal_properties, 'openmember_properties', None)
        if openmember_properties.hasProperty('title_fields'):
            return list(openmember_properties.getProperty('title_fields'))
        else:
            return []
    
    def set_title_fields(self, value):
        """The address fields are stored in the properties tool, on the openmember_properties sheet lines field called address_fields.
        """
        portal_properties = getToolByName(self.context, 'portal_properties', None)
        openmember_properties = getattr(portal_properties, 'openmember_properties', None)
        openmember_properties.manage_changeProperties(title_fields=tuple(value))
        
    title_fields = property(get_title_fields,
                            set_title_fields)
    
registerForm(IOMControlPanelDatabaseSchema)


class OMControlPanel(ControlPanelForm):
    """Base class for the control panel.
    When rendering it uses all the sub control panels registered using the 
    betahaus.openmember.browser.controlpanel.registerForm method as sub panels.
    """
        
    def __init__(self, context, request):
        super(OMControlPanel, self).__init__(context, request)
        forms = []
        for interface in _controlpanel_forms:
            if Interface.providedBy(interface) and IOMControlPanelForm in interface.getBases():
                adapter = interface(self.context)
                form = FormFieldsets(interface)
                form.id = adapter.getId()
                form.label = adapter.getLabel()
                forms.append(form)
            
        self.form_fields = FormFieldsets(*forms)
        
        
    label = _(u"Open Member control panel")
    description = _(u"Tools and settings for Open Member database.")

    def available(self):
        root = aq_inner(self.context).getPhysicalRoot()
        sm = getSecurityManager()
        return sm.checkPermission(OPENMEMBER_MANAGE_DATABASE, root)

    @form.action(_(u'label_save', default=u'Save'), name=u'save')
    def handle_edit_action(self, action, data):
        CheckAuthenticator(self.request)
        if form.applyChanges(self.context, self.form_fields, data,
                             self.adapters):
            self.status = _("Changes saved.")
            notify(ConfigurationChangedEvent(self, data))
        else:
            self.status = _("No changes made.") #Does this work? No Idea?

    @form.action(_(u'label_cancel', default=u'Cancel'),
                 validator=null_validator,
                 name=u'cancel')
    def handle_cancel_action(self, action, data):
        IStatusMessage(self.request).addStatusMessage(_("Changes canceled."),
                                                      type="info")
        url = getMultiAdapter((self.context, self.request),
                              name='absolute_url')()
        self.request.response.redirect(url + '/plone_control_panel')
        return ''
        
    def _on_save(self, data=None):
        pass

    @form.action(_(u'Clear and rebuild catalog'), name=u'clear_and_rebuild')
    def clear_and_rebuild(self, action, data):
        CheckAuthenticator(self.request)
        if not self.available():
            self.status = _(u'text_not_allowed_manage_server',
                            default=u'You are not allowed to manage the Zope server.')        
            return
        catalog = getToolByName(self.context, 'openmember_catalog')
        catalog.clearFindAndRebuild()
        self.status = _(u'Clear and rebuild complete')
        
    @form.action(_(u'Revisist members'), name=u'revisit_members')
    def revisit_members(self, action, data):
        CheckAuthenticator(self.request)
        if not self.available():
            self.status = _(u'text_not_allowed_manage_server',
                            default=u'You are not allowed to manage the Zope server.')        
            return
        catalog = getToolByName(self.context, 'openmember_catalog')
        catalog.revisit_mirrored_objects()
        self.status = _(u'Revisit of members complete')
    
        
