import array
import base64
from .generator import Generator
import krpc.types

class CsharpGenerator(Generator):

    def __init__(self, macro_template, service, definition_files):
        super(CsharpGenerator, self).__init__(macro_template, service, definition_files)

    _keywords = set([
        'abstract', 'as', 'base', 'bool', 'break', 'byte', 'case', 'catch', 'char', 'checked', 'class',
        'const', 'continue', 'decimal', 'default', 'delegate', 'do', 'double', 'else', 'enum', 'event',
        'explicit', 'extern', 'false', 'finally', 'fixed', 'float', 'for', 'foreach', 'goto', 'if',
        'implicit', 'in', 'int', 'interface', 'internal', 'is', 'lock', 'long', 'namespace', 'new',
        'null', 'object', 'operator', 'out', 'override', 'params', 'private', 'protected', 'public',
        'readonly', 'ref', 'return', 'sbyte', 'sealed', 'short', 'sizeof', 'stackalloc', 'static',
        'string', 'struct', 'switch', 'this', 'throw', 'true', 'try', 'typeof', 'uint', 'ulong',
        'unchecked', 'unsafe', 'ushort', 'using', 'virtual', 'void', 'volatile', 'while'
    ])

    def parse_name(self, name):
        if name in self._keywords:
            return '%s_' % name
        else:
            return name

    def parse_type(self, typ):
        if isinstance(typ, krpc.types.ValueType):
            typ = typ.protobuf_type
            if typ == 'string':
                return 'String'
            elif typ == 'bytes':
                return 'byte[]'
            elif typ == 'float':
                return 'Single'
            elif typ == 'double':
                return 'Double'
            elif typ == 'bool':
                return 'Boolean'
            elif 'int' in typ:
                int_type_map = {
                    'int16' : 'Int16',
                    'uint16': 'UInt16',
                    'int32' : 'Int32',
                    'uint32': 'UInt32',
                    'int64' : 'Int64',
                    'uint64': 'UInt64'
                }
                return int_type_map[typ]
        elif isinstance(typ, krpc.types.MessageType):
            typ = typ.protobuf_type
            if typ.startswith('KRPC.'):
                _, _, x = typ.rpartition('.')
                return 'global::KRPC.Schema.KRPC.%s' % x
            elif typ.startswith('Test.'):
                _, _, x = typ.rpartition('.')
                return 'global::Test.%s' % x
        elif isinstance(typ, krpc.types.ListType):
            return 'global::System.Collections.Generic.IList<%s>' % \
                self.parse_type(self.types.as_type(typ.protobuf_type[5:-1]))
        elif isinstance(typ, krpc.types.SetType):
            return 'global::System.Collections.Generic.ISet<%s>' % \
                self.parse_type(self.types.as_type(typ.protobuf_type[4:-1]))
        elif isinstance(typ, krpc.types.DictionaryType):
            key_type, value_type = tuple(typ.protobuf_type[11:-1].split(','))
            return 'global::System.Collections.Generic.IDictionary<%s,%s>' % \
                (self.parse_type(self.types.as_type(key_type)), self.parse_type(self.types.as_type(value_type)))
        elif isinstance(typ, krpc.types.TupleType):
            value_types = typ.protobuf_type[6:-1].split(',')
            return 'global::System.Tuple<%s>' % ','.join(self.parse_type(self.types.as_type(t))
                                                         for t in value_types)
        elif isinstance(typ, krpc.types.ClassType):
            return 'global::KRPC.Client.Services.%s' % typ.protobuf_type[6:-1]
        elif isinstance(typ, krpc.types.EnumType):
            return 'global::KRPC.Client.Services.%s' % typ.protobuf_type[5:-1]
        raise RuntimeError('Unknown type ' + typ)

    def parse_return_type(self, procedure):
        if 'return_type' in procedure is not None:
            typ = self.types.get_return_type(procedure['return_type'], procedure['attributes'])
            return self.parse_type(typ)
        else:
            return 'void'

    def parse_parameter_type(self, typ):
        return self.parse_type(typ)

    def parse_default_value(self, value, typ):
        value = base64.b64decode(value)
        #TODO: following is a workaround for decoding EnumType, as set_values has not been called
        value = array.array('B', value).tostring()
        if not isinstance(typ, krpc.types.EnumType):
            value = krpc.decoder.Decoder.decode(value, typ)
        else:
            value = krpc.decoder.Decoder.decode(value, self.types.as_type('int32'))
        if isinstance(typ, krpc.types.ValueType) and typ.protobuf_type == 'string':
            return '"%s"' % value
        if isinstance(typ, krpc.types.ValueType) and typ.protobuf_type == 'bool':
            if value:
                return 'true'
            else:
                return 'false'
        elif isinstance(typ, krpc.types.ValueType) and typ.protobuf_type == 'float':
            return str(value) + "f"
        elif isinstance(typ, krpc.types.ClassType) and value is None:
            return 'null'
        elif isinstance(typ, krpc.types.EnumType):
            return '(global::KRPC.Client.Services.%s)%s' % (typ.protobuf_type[5:-1], value)
        else:
            return value

    @staticmethod
    def parse_documentation(documentation):
        documentation = documentation.replace('<doc>', '').replace('</doc>', '').strip()
        if documentation == '':
            return ''
        lines = ['/// '+line for line in documentation.split('\n')]
        content = '\n'.join(line.rstrip() for line in lines)
        content = content.replace('  <param', '<param')
        content = content.replace('  <returns', '<returns')
        content = content.replace('  <remarks', '<remarks')
        return content

    @staticmethod
    def parse_context(context):
        return context
