# Post processing set of tools for the coastal vulnerability model.
# Currently implemented functions (functional, but not stable yet):
# - convert_tif_to_csv
# - convert_tifs_to_csv
# - aggregate_csvs
# - aggregate_tifs_from_directory
# See at the end of this file for examples of how to use these functions
import os

import csv
import numpy as np
from osgeo import gdal

def execute(args):
    target_directory = args['workspace_dir']
    mask = args['shore_segment_mask'].encode('utf-8') \
        if 'shore_segment_mask' in args else None

    aggregate_tifs_from_directory(path = target_directory, mask = mask)

def convert_tif_to_csv(tif_uri, csv_uri = None, mask = None):
    """ Converts a single band geo-tiff file to a csv text file
    
        Inputs:
            -tif_uri: the uri to the file to be converted
            -csv_uri: uri to the output file. The file should not exist.

        Outputs:
            -returns the ouput file uri

        returns nothing"""
    # The file has to exist
    assert(os.path.isfile(tif_uri))
    # The file has to be a tif
    extension = os.path.splitext(tif_uri)[1][1:].strip().lower()
    assert(extension == 'tif')
    # Create a uri for the new csv, and ensure it doesn't already exist
    if csv_uri == None:
        csv_uri = os.path.splitext(tif_uri)[0] + '.csv'
    assert(not os.path.exists(csv_uri))

    print("converting '" + tif_uri + "' to '" + csv_uri + "'...")
    dataset = gdal.OpenEx(tif_uri)
    assert(dataset)
    array = dataset.GetRasterBand(1).ReadAsArray()
    gt = dataset.GetGeoTransform()
    origin = np.array([gt[3], gt[0]])
    step = np.array([gt[5], gt[1]])
    size = np.array([dataset.RasterYSize, dataset.RasterXSize])
    I = np.array(range(size[0])) * step[0] + origin[0]
    J = np.array(range(size[1])) * step[1] + origin[1]
    
    np.set_printoptions(precision = 20)
    with open(csv_uri, 'wb') as csv_file:
        writer = csv.writer(csv_file)
        #row = ['x', 'y', 'row', 'col', 'value']
        # TODO: use the filename instead of 'value' for the column ID
        row_name = '_'.join(csv_uri.split('.')[0].split('_')[-3:])
        row = ['x', 'y', row_name]
        writer.writerow(row)
        if mask:
            for index in range(mask[0].size):
                i = mask[0][index]
                j = mask[1][index]
                row = [J[j], I[i], array[i, j]]
                writer.writerow(row)
        else:
            for i in range(size[0]):
                #print('writing row ' + str(i) + '/' + str(size[0]))
                for j in range(size[1]):
                    #row = [J[j], I[i], i, j, array[i, j]]
                    row = [J[j], I[i], array[i, j]]
                    writer.writerow(row)
    return csv_uri

def convert_tifs_to_csv(tif_list, mask = None):
    csv_list = []
    for tif_uri in tif_list:
        csv_list.append(convert_tif_to_csv(tif_uri, mask = mask))
    return csv_list

def aggregate_csvs(csv_list, out_uri):
    """Concatenate 3-row csv files created with tif2csv
        
        Inputs:
            - csv_list: list of csv_uri strings
            
        Outputs:
            - uri_output: the output uri of the concatenated csv"""
    assert(os.path.exists(csv_list[0]))
    X = []
    Y = []
    values = []
    with open(csv_list[0]) as csv_file:
        reader = csv.reader(csv_file)
        reader.next()
        for row in reader:
            X.append(row[0])
            Y.append(row[1])

    aggregated_data = []
    for csv_uri in csv_list:
        assert(os.path.exists(csv_uri))
        print('reading ' + csv_uri)
        data = []
        with open(csv_uri) as csv_file:
            reader = csv.reader(csv_file)
            header = reader.next()
            values.append(header[2])
            for row in reader:
                data.append(row[2])
        aggregated_data.append(data)

    np.set_printoptions(precision = 20)
    with open(out_uri, 'wb') as csv_file:
        writer = csv.writer(csv_file)
        row = ['x', 'y']
        for name in values:
            row.append(name)
        writer.writerow(row)
        print('writing ' + str(len(X)) + " rows in '" + out_uri + "'...")
        for i in range(len(X)):
            #print('writing row ' + str(i) + '/' + str(len(X)))
            row = [X[i], Y[i]]
            for value in aggregated_data:
                row.append(value[i])
            writer.writerow(row)
    return csv_uri

def aggregate_tifs_from_directory(path = '.', mask = None):
    if mask:
        assert(os.path.exists(mask))
        raster = gdal.OpenEx(mask)
        band = raster.GetRasterBand(1)
        nodata = band.GetNoDataValue()
        array = band.ReadAsArray()
        mask = np.where(array != nodata)
    
    assert(os.path.exists(path))
    base_dir = os.getcwd()
    os.chdir(path)

    uri_list = os.listdir('.')
    tif_list = []
    for uri in uri_list:
    	if '.tif' in uri:
            tif_list.append(uri)
    tif_count = len(tif_list)
    if not tif_count:
        print('No tifs found in', path)
        return
    dir_name = os.getcwd().split(os.sep)[-1]
    print('dir_name', dir_name)
    out_uri = os.path.join(os.getcwd(), dir_name + '.csv')
    print('aggregating ' + str(len(tif_list)) + ' tifs from ' + \
        path + ' to ' + out_uri)
    csv_list = convert_tifs_to_csv(tif_list, mask = mask)
    aggregate_csvs(csv_list, out_uri)
    for csv in csv_list:
        os.remove(csv)
    
    return out_uri

# Usage example:
# convert one single tiff file to csv:
#csv_uri = convert_tif_to_csv('example.tif')
#print('csv file uri', csv_uri) # prints 'example.csv'
# convert a list of tiff files to csv:
#csv_uri_list = convert_tifs_to_csv(['1.tif', '2.tif'])
#print('csv list', csv_uri_list) # prints ['1.csv', '2.csv']
# aggregate a list of csvs to a single file:
#out_uri = aggregate_csvs(convert_tifs_to_csv( \
#    ['1.tif', '2.tif', '3.tif', '4.tif', '5.tif']), 
#    'big_fat_file.csv')
# aggregate tiffs in the current directory to a single csv named after the current directory:
#aggregate_tifs_from_directory()
# aggregate tiffs from the subdirectory '../test_dir' to a csv named 'subdir.csv':
#aggregate_tifs_from_directory( \
#    path = '../../test/data/test_out/coastal_vulnerability/outputs/fetch', \
#    mask = '../../intermediate/00_PRE_shore.tif')
