""" Coastal vulnerability model core functions """
import os
import csv
import re
import math
import shutil
import glob

import numpy as np
import scipy as sp
from scipy import spatial
from scipy.ndimage import morphology
from rtree import index
import shapely
import shapely.wkb

from osgeo import gdal
from osgeo import ogr
from osgeo import osr

import logging
import natcap.invest.pygeoprocessing_0_3_3.geoprocessing


logging.getLogger("natcap.invest.pygeoprocessing_0_3_3.geoprocessing").setLevel(logging.WARNING)
logging.getLogger("raster_cython_utils").setLevel(logging.WARNING)
LOGGER = logging.getLogger('natcap.invest.coastal_vulerability.core')

SECTOR_COUNT = 16 # Number of equi-angular sectors
MAX_FETCH = 60000 # Longest fetch ray
SHELTERED_SHORE = 0
EXPOSED_SHORE = 1

def adjust_dataset_ranks(input_uri, output_uri):
    """Adjust the rank of a dataset's first band using 'adjust_layer_ranks'.

        Inputs:
            - input_uri: dataset uri where values are 1, 2, 3, 4, or 5
            - output_uri: new dataset with values adjusted by
              'adjust_layer_ranks'.

        Returns output_uri."""
    # Open the raster
    input_raster = gdal.OpenEx(input_uri)
    assert input_raster is not None
    # Compute adjusted rank values
    band = input_raster.GetRasterBand(1)
    rank_data = band.ReadAsArray()
    adjusted_ranks = adjust_layer_ranks(rank_data)
    # Compute nodata--ensure it's of compatible type
    new_nodata = 0.
    # Create the new raster with appropriate data in it
    output_raster = natcap.invest.pygeoprocessing_0_3_3.geoprocessing.new_raster_from_base(input_raster, \
        output_uri, 'GTiff', new_nodata, gdal.GDT_Byte)
    # Save the array in the raster
    band = output_raster.GetRasterBand(1)
    band.WriteArray(adjusted_ranks)
    # Done. Return the new data as a numpy array.
    return output_uri

def adjust_layer_ranks(layer):
    """Adjust the rank of a layer in case there are less than 5 values.

        Inputs:
            - layer: a float or int numpy array as extracted by ReadAsArray
                that encodes the layer ranks (valued 1, 2, 3, 4, or 5).

        Output:
            - adjusted_layer: a numpy array of same dimensions as the input
              array with rank values reassigned follows:
                -non-shore segments have a (no-data) value of zero (0)
                -all segments have the same value: all are set to a rank of 3
                -2 different values: lower values are set to 3, 4 for the rest
                -3 values: 2, 3, and 4 by ascending level of vulnerability
                -4 values: 2, 3, 4, and 5 by ascending level of vulnerability
    """
    # Adjusted ranks as described in Greg's notes, listed in ascending order
    adjusted_ranks = {}
    adjusted_ranks[0] = [1., 2., 3., 4., 5.] # Used to validate input
    adjusted_ranks[1] = [3.]
    adjusted_ranks[2] = [3., 4.]
    adjusted_ranks[3] = [2., 3., 4.]
    adjusted_ranks[4] = [2., 3., 4., 5.]
    # Find unique values, sorted in ascending order, as are the adjusted ranks
    unique_values = np.unique(layer)
    allowed_values = np.array(adjusted_ranks[0])
    nodata = np.setdiff1d(unique_values, allowed_values)
    if len(nodata) >= 1:
        LOGGER.warning('Multiple nodata values (values <= 0) detected, '
                       'setting all to 0.')
        layer[layer < 0.] = 0.0

    valid_values = np.setdiff1d(unique_values, nodata)
    # Every element has to be valid, i.e. all elements in adjusted_ranks are in
    # unique
    # If maximum number of values, no need to do anything
    if len(valid_values) == len(adjusted_ranks):
        return layer
    # If less than maximum values, create the return array...
    adjusted_layer = np.copy(layer)
    # ...and replace each rank with the predefined ones in adjusted_ranks
    for value in range(valid_values.size):
        input_value = valid_values[value]
        adjusted_value = adjusted_ranks[valid_values.size][value]
        adjusted_layer[layer == input_value] = adjusted_value

    return adjusted_layer

def execute(args):
    """ Entry point for coastal vulnerability core

        args[''] - actual data structure the way I want them look like  :RICH:DESCRIBE ALL THE ARGUMENTS IN ARGS


        returns nothing"""
    # Indices in the layer info list
    # TODO: Change the list to a dictionary
    PREFIX = 0 # String concatenated to every file generated
    KEY = 1 # Output name, used to construct other variables (output, constant)
    OUTPUT_TYPE = 2 # String used to discriminate between layer and non-layer
                    # output, but can be used for any other purpose
    PRECONDITIONS = 3 # List of keys in args required to compute a given output
    FUNCTION = 4 # Function callback that computes a particular output

    # outputs_info: A list containing information about the way the outputs
    # will be stored on disk. Each entry in the list is a computational step,
    # and each element in the entry follows the same convention:
    # 0- Prefix: a string concatenated to every file/directory generated
    # 1- Key: short string describing the type of computation performed for
    # that step. Used to create other variables.
    # 2- Output type: used as a flag for re-classifying data using
    # adjust_layer_ranks
    # 3- preconditions: 2-level nested string list of keys expected in the args
    # dictionary before initiating the computation of that step.
    # The args dictionary is searched for the strings in the inner lists. An
    # inner list is evaluated to 'True' if all its strings are found in args.
    # The computation is initiated if at least one inner list is 'True'
    # 4- Function callback used for that step
    outputs_info = [ \
    ['00_', 'preprocessing'], \
    ['0_', 'structures', 'keep',[['structures_uri']], \
        compute_structure_protection], \
    ['1_a_', 'segment_exposure', 'keep', [['fetch_distances', \
        'fetch_depths']], compute_segment_exposure],\
    ['1_b_', 'geomorphology', 'keep', [['geomorphology_uri']], \
        compute_geomorphology], \
    ['1_c_', 'relief', 're-classify',[['relief_uri']], compute_relief_rank], \
    ['1_d_', 'natural_habitats', 'keep', [['habitats_csv_uri']], \
        compute_natural_habitats_vulnerability], \
    ['1_e_', 'wind_exposure','re-classify',[['climatic_forcing_uri', \
        'fetch_distances']],\
        compute_wind_exposure],\
    ['1_f_', 'wave_exposure', 're-classify', [['climatic_forcing_uri', \
        'fetch_depths']],\
        compute_wave_exposure],\
    ['1_g_', 'surge_potential', 're-classify', [['depth_contour', \
    'bathymetry_raster_uri'], ['continental_shelf_raster_uri']], \
        compute_surge_potential],\
    ['1_h_', 'sea_level_rise', 're-classify', [['sea_level_rise_uri']], \
        compute_sea_level_rise],\
    ['1_i_','coastal_exposure','keep',[['geomorphology'], \
    ['relief'], ['surge_potential'], ['sea_level_rise'], ['natural_habitats'],\
    ['wave_exposure'], ['optional_layer']], \
        compute_coastal_exposure], \
    ['1_j_', 'coastal_exposure_no_habitats', 'keep',
        [['natural_habitats', 'coastal_exposure']], \
        compute_coastal_exposure_no_habitats],\
    ['1_k_', 'habitat_role', 'keep', [['coastal_exposure_no_habitats']],\
        compute_habitat_role],\
    ['1_l_', 'erosion_exposure', 'keep', \
    [['geomorphology', 'natural_habitats', 'wave_exposure','sea_level_rise']],\
        compute_erosion_exposure],\
    ['1_m_', 'population_map', 'keep', [['population_uri']], \
        compute_coastal_population],\
    ['1_n_', 'additional_layer', 're-classify', [['additional_layer_uri']], \
        compute_additional_layer],\
    ['2_', 'fetch','keep',[['fetch_distance_uris'],['fetch_depth_uris']],\
        save_fetch_to_outputs],\
    ['3_1_', 'structure_edges', 'keep', [['structure_edges']], \
        save_structure_to_subdirectory],\
    ['3_2_', 'erodible_shoreline', 'keep', [['geomorphology']],\
        compute_erodible_shoreline],\
    ['3_3_', 'continental_shelf_distance', 'keep', \
        [['shore_shelf_distance']],\
        compute_continental_shelf_distance],\
    ['3_5_', 'wind_generated_waves', 'keep', \
        [['wave_heights', 'wave_periods']], \
        save_wind_generated_waves_to_subdirectory],\
    ['3_6_', 'oceanic_wave_exposure', 'keep', [['E_o']], \
        save_oceanic_wave_exposure_to_subdirectory],\
    ['3_7_', 'local_wave_exposure', 'keep', [['E_l']], \
        save_local_wave_exposure_to_subdirectory]]

    args['outputs_info'] = outputs_info

    # Each intermediate file is copied to its directory based on its prefix.
    for index in range(1, len(outputs_info)):
        # All the info about this output (list)
        output = outputs_info[index]
        key = output[KEY]
        output_name = key.replace('_', ' ')
        # Add fields to the function arguments
        args['prefix'] = output[PREFIX]
        args['subdirectory'] = args['prefix'] + key
        # Check that the subdirectory exists
        subdirectory = \
            os.path.join(args['intermediate_directory'], args['subdirectory'])
        if not os.path.exists(subdirectory):
            os.makedirs(subdirectory)
        # Filename in /outputs
        output_file = output[PREFIX] + key + '.tif'
        output_path = os.path.join(subdirectory, output_file)
        # Dictionary key to the output path
        output_key = key + '_output'
        # Dictionary key to the user-defined output constant
        output_constant = key + '_constant'
        # Check preconditions and list the ones missing
        preconditions_met = np.array([
            np.array([keyword in args for keyword in inner_list]).all()
            for inner_list in output[PRECONDITIONS]])
        if preconditions_met.any():
            LOGGER.info('Processing %s ...', output_name)
            # Compute the ranks for the output, and save in data_uri
            fn_output = output[FUNCTION](args)

            # ----------------
            # Debug, just to make sure no file gets accidentally created.
            entries = glob.glob(os.path.join(args['intermediate_directory'], \
                '*.tif'))
            entries_count = len(entries)
            #if entries_count:
            #    print('total entries', len(entries))
            #    for entry in entries:
            #        print entry
            #    assert len(entries) == 0
            #print('added keys to args:')
            #for k in fn_output.keys():
            #    print(k)
            # ----------------

            # Make sure the new entries won't erase anything in args:
            args_keys = set(args.keys())
            new_keys = set(fn_output.keys())
            intersection = args_keys.intersection(new_keys)
            if intersection:
                error_message = "Output from " + output_name + \
                    ' overlaps with ' + str(intersection) + '.'
                raise RuntimeError(error_message)
            # Reclassify ranks if necessary:
            if output[OUTPUT_TYPE] == 're-classify':
                # Adjust the ranks, save in outputs and save the uri in args
                #LOGGER.info('reclassifying ' + key)
                fn_output[output_key] = \
                    adjust_dataset_ranks(fn_output[key], output_path)
            else:
                assert output[OUTPUT_TYPE] == 'keep'
            # Add the final outputs to args:
            args = \
                dict(args.items() + fn_output.items())
        else:
            # Preconditions are not met. Print the ones that are missing.
            missing_preconditions = []
            for inner_list in output[PRECONDITIONS]:
                for keyword in inner_list:
                    if keyword not in args:
                        missing_preconditions.append(keyword)
            LOGGER.debug(output_name + ' not loaded. Missing key:' + \
                str(missing_preconditions))
            #LOGGER.debug('Available keys are:')
            #for key in sorted(args.keys()):
            #    print(key)
            # Assign a constant value to the entire layer if it is specified
            if output_constant in args:
                constant = int(args[output_constant])
                shore_dataset = gdal.OpenEx(args['shore_raster_uri'])
                shore_array = shore_dataset.GetRasterBand(1).ReadAsArray()
                LOGGER.info('Assigning %s to %s', constant, output_name)
                raster = natcap.invest.pygeoprocessing_0_3_3.geoprocessing.new_raster_from_base( \
                    gdal.OpenEx(args['aoi_raster_uri']), output_file, \
                    'GTIFF', 0.0, gdal.GDT_Float32)
                raster.GetRasterBand(1).WriteArray(shore_array * constant)
                args[output_key] = output_path
            # Skip the layer entirely
            else:
                print('Skipping ' + output_name + '.')

    # The model groups output files in sub-directories.
    # The information about the groups (which files should be in a group)
    # that are going to be created is organized in a dictionary
    # where each entry describes a group:
    #   -key is the group name,
    #   -value is a list of string tuples that tell how to look for uris:
    #       -tuple[0]: uri selection method. Can be either one of:
    #           -include: add uris which keys include the token in tuple[1]
    #           -match: add only uris which key exactly matches tuple[1]
    #       -tuple[1]: string used to select uris
    # A file can be in more than one group.
    output_groups = { \
        "coastal_exposure": [('match', "shore_exposure"), \
            ('match', "geomorphology"), \
            ('match', "relief"), ('match', "natural_habitats"), \
            ('match', "sea_level_rise"), \
            ('match', "wave_exposure"), \
            ('match', "surge_potential"), ('match', "erodible_shoreline"), \
            ('match', "additional_layer_raw", "additional_layer_ranked"), \
            ('match', "coastal_exposure"), \
            ('match', "coastal_exposure_no_habitats"), \
            ('match', "habitat_role")],
        "population": [('match', 'shore_exposure'), \
            ('match', 'coastal_population'), ('match', 'structure_edges'), \
            ('match', 'erodible_shoreline'), \
            ('match', 'coastal_exposure'), \
            ('match', 'coastal_exposure_no_habitats')] \
        }

    # Create groups in the outputs/ folder.
    print('creating groups...')
    for name in output_groups.keys():
        # Populate the group with the keys in args
        group_keys = []
        for token_info in output_groups[name]:
            selection_operator = token_info[0]
            file_token = token_info[1]
            # For each token, create a list of uris sorted alphabetically
            sub_group = []
            if selection_operator == 'include':
                for item in args.items():
                    if (file_token in item[0]) and \
                        (type(item[1]) == str) and ('.tif' in item[1]):
                        sub_group.append(item[0])
            elif selection_operator == 'match':
                for item in args.items():
                    if (file_token == item[0]) and \
                        (type(item[1]) == str) and ('.tif' in item[1]):
                        sub_group.append(item[0])
                #if not sub_group:
                    #message = "Can't include " + file_token + ' in group ' + \
                    #    name + '. Existing tokens are: ' + \
                    #    str(sorted(args.keys()))
                    #LOGGER.warning(message)
            else:
                message = 'Invalid file selection operator ' + \
                selection_operator + '. Should be either "include" or "match"'
                assert False, message
            sub_group.sort()
            # Add the sorted list to the group
            group_keys.extend(sub_group)
        group_size = len(group_keys)
        print('group ' + name + ' size ' + str(group_size))
        # The order of uris should match the keys order
        group_uris = []
        for i in range(group_size):
            group_uris.append(args[group_keys[i]])
        # Create the group directories
        directory_uri = os.path.join(args['outputs_directory'], name)
        if not os.path.exists(directory_uri):
            os.makedirs(directory_uri)
        # Step 1, build a list of uris:
        output_uri_list = [os.path.join(directory_uri, os.path.basename(uri)) \
        for uri in group_uris]
        # Post-process the output files:
        # -clip and align the data so that the rasters are all the same size
        interpolation_mode = ['nearest'] * group_size
        natcap.invest.pygeoprocessing_0_3_3.geoprocessing.align_dataset_list(group_uris, output_uri_list,
        interpolation_mode, args['cell_size'], 'intersection', 0)
        # -mask out the exposed shore segments if we only want protected ones
        if args['area_computed'] == 'sheltered':
            LOGGER.info('Masking exposed segments for:')
            def mask_exposed_segments(exposure, x):
                return 0. if exposure == EXPOSED_SHORE else x
            # Create temporary URIs so that vectorize_datasets doesn't override
            # its own inputs, which would create data corruption
            tmp_uri_list = \
            [natcap.invest.pygeoprocessing_0_3_3.geoprocessing.temporary_filename() for _ in output_uri_list]
            for tmp_uri, out_uri in zip(tmp_uri_list, output_uri_list):
                if 'shore_exposure' in out_uri:
                    continue
                LOGGER.info(out_uri)
                natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets( \
                [args['shore_exposure'], out_uri], mask_exposed_segments, \
                tmp_uri, gdal.GDT_Float32, 0., args['cell_size'], \
                'intersection')
                # Copy the final result over to the output URI
                shutil.copy(tmp_uri, out_uri)
        #print('output uri list:')
        for i in range(len(output_uri_list)):
            out_uri_raster = gdal.OpenEx(output_uri_list[i])
            out_uri_shape = out_uri_raster.GetRasterBand(1).ReadAsArray().shape
            out_uri_raster = None
            #print(out_uri_shape, output_uri_list[i])


    # Do some post-processing (export each group's shore information
    # to a csv file)
    mask_uri = args['shore_raster_uri']
    base_path = os.path.join(os.getcwd(), args['outputs_directory'])
    for group_name in output_groups.keys():
        #print('group_name', group_name)
        aggregate_tifs_from_directory( \
            path = os.path.join(base_path, group_name), \
            mask = mask_uri)


    # Create CSVs in the intermediate directories
    # So far, it only creates a CSV in one directory, which is why this looks
    # hacky. The function aggregate_tifs might break if the rasters are not the
    # same size. In this case, they can be clipped with
    # natcap.invest.pygeoprocessing_0_3_3.geoprocessing.clip_dataset_list
    processed_dir_list = \
    [
    os.path.join(args['intermediate_directory'], '1_a_segment_exposure'), \
    os.path.join(args['intermediate_directory'], '1_b_geomorphology'), \
    os.path.join(args['intermediate_directory'], '1_c_relief'), \
    os.path.join(args['intermediate_directory'], '1_d_natural_habitats'), \
    os.path.join(args['intermediate_directory'], '1_e_wind_exposure'), \
    os.path.join(args['intermediate_directory'], '1_f_wave_exposure'), \
    os.path.join(args['intermediate_directory'], '1_g_surge_potential'), \
    os.path.join(args['intermediate_directory'], '1_h_sea_level_rise'), \
    os.path.join(args['intermediate_directory'], '1_i_coastal_exposure'), \
    os.path.join(args['intermediate_directory'], \
        '1_j_coastal_exposure_no_habitats'), \
    os.path.join(args['intermediate_directory'], '1_k_habitat_role'), \
    os.path.join(args['intermediate_directory'], '1_l_erosion_exposure'), \
    os.path.join(args['intermediate_directory'], '1_m_population_map'), \
    os.path.join(args['intermediate_directory'], '1_n_additional_layer'), \
    os.path.join(args['intermediate_directory'], '2_fetch'), \
    os.path.join(args['intermediate_directory'], '3_1_structure_edges'), \
    os.path.join(args['intermediate_directory'], '3_2_erodible_shoreline'), \
    os.path.join(args['intermediate_directory'], \
        '3_3_continental_shelf_distance'), \
    os.path.join(args['intermediate_directory'], '3_5_wind_generated_waves'), \
    os.path.join(args['intermediate_directory'], '3_6_oceanic_wave_exposure'), \
    os.path.join(args['intermediate_directory'], '3_7_local_wave_exposure')
    ]
    token_list = [ \
    ['long', 'open', 'deep', 'average', 'pct', 'bool', 'exposure'], \
    ['geomorphology'], \
    ['relief', 'average_land_height'], \
    ['natural_habitats'], \
    ['REI', 'wind_exposure'], \
    ['wave_height', 'wave_period', 'oceanic_wave', 'local_wave'], \
    ['surge_potential', 'shore_shelf_distance'], \
    ['sea_level_rise', ], \
    ['coastal_exposure'], \
    ['coastal_exposure_no_habitats'], \
    ['habitat_role'], \
    ['structures'], \
    ['coastal_population'], \
    ['n_additional_layer', 'additional_layer_raw'], \
    ['fetch_depth', 'fetch_distance'], \
    ['structure_edges'], \
    ['erodible_shoreline'], \
    ['continental_shelf'], \
    ['wave_height', 'wave_period'], \
    ['oceanic_wave_exposure'], \
    ['local_wave_exposure'] \
    ]
    # TODO: Add habitat data used in the computation of habitats
    for processed_dir, tokens_to_match in zip(processed_dir_list, token_list):
#        print('processed_dir', processed_dir)
#        print('tokens_to_match', tokens_to_match)

        dir_content = os.listdir(processed_dir)
        files_to_process = []
        for f in dir_content:
#            print('  file under consideration:', f)
            for t in tokens_to_match:
#                print('  Token to match:', t)
                if t in f:
#                    print('    Match!')
                    files_to_process.append(os.path.join(processed_dir, f))
                    break
        files_to_process = sorted(files_to_process)
#        print('files list')
#        for f in files_to_process:
#            print(os.path.basename(f))
        aggregate_tifs_from_list(files_to_process, processed_dir, \
            mask = mask_uri)

    # Create point shapefile from coastal_exposure CSV file
    csv_path = os.path.join(
        args['workspace_dir'], 'outputs', 'coastal_exposure',
        'coastal_exposure.csv')
    out_path = os.path.join(
        args['workspace_dir'], 'outputs', 'coastal_exposure',
        'coastal_exposure.shp')
    spat_ref = natcap.invest.pygeoprocessing_0_3_3.geoprocessing.get_spatial_ref_uri(
        args['aoi_uri'])
    # A dictionary mapping coastal exposure CSV column headers
    # to truncated / abbreviated versions to use as field in Shapefile
    header_map = {'x': 'x', 'y': 'y', 'array row': 'array_row',
        'array col': 'array_col', 'shore_exposure': 'shore_exp',
        'geomorphology': 'geomorph', 'relief': 'relief',
        'natural_habitats': 'nat_habs', 'wave_exposure': 'wave_exp',
        'surge_potential': 'surge_pot', 'coastal_exposure': 'ExpIndex',
        'coastal_exposure_no_habitats': 'ExpIndNoHb',
        'habitat_role': 'hab_role', 'erodible_shoreline': 'erod_shore'}
    # Open CSV file for reading and create dictionary with 'x', 'y' as keys
    # Any fields NOT defined in header_map will be skipped.
    csv_file = open(csv_path, 'rU')
    reader = csv.DictReader(csv_file)
    csv_dict = {}
    # Get column fields to preserve order in creating shapefile
    columns = [header_map[col] for col in reader.fieldnames
               if col in header_map]
    # List to store (x,y) keys to preserve order of rows in shapefile
    row_order = []
    for row in reader:
        row_order.append((row['x'], row['y']))

        # Build up the CSV row dict based on the header_map dict defined
        # above.  Any fields NOT defined in header_map will be skipped.
        csv_dict[(row['x'], row['y'])] = dict(
            (header_map[key], val) for (key, val) in row.iteritems()
            if key in header_map)

    dict_to_point_shapefile(
        csv_dict, out_path, spat_ref, columns, row_order)

    # Create url shortcut to CV dashboard, replacing the need to
    # generate a html summary report
    html_path = os.path.join(
        args['workspace_dir'], 'outputs',
        'vulpes.sefs.uw.edu-ttapp-cv-dash.php.url')
    html_file = open(html_path, 'w')
    html_file.write(
        "[InternetShortcut]\n"
        "URL=http://vulpes.sefs.uw.edu/ttapp/cv-dash.php")
    html_file.close()

    return

def dict_to_point_shapefile(dict_data, out_path, spat_ref, columns, row_order):
    """Create a point shapefile from a dictionary.

    Parameters:
        dict_data (dict): a dictionary where keys point to a sub
            dictionary that has at least keys 'x', 'y'. Each
            sub dictionary will be added as a point feature using 'x', 'y'
            as the geometry for the point. All other key, value pairs in
            the sub dictionary will be added as fields and values to the point
            feature.
        out_path (string): a path on disk for the point shapefile.
        spat_ref (osr spatial reference): an osr spatial reference to
            use when creating the layer.
        columns (list): a list of strings representing the order the
            field names should be written. Attempting the attribute table
            reflects this order.
        row_order (list): a list of tuples that match the keys of
            'dict_data'. This is so we can add the points in a specific order
            and hopefully populate the attribute table in that order.

    Returns:
        Nothing
    """

    # If 'out_path' exists delete it
    if os.path.isfile(out_path):
        driver = ogr.GetDriverByName('ESRI Shapefile')
        driver.DeleteDataSource(out_path)

    LOGGER.debug('Creating new datasource')
    driver = ogr.GetDriverByName('ESRI Shapefile')
    datasource = driver.CreateDataSource(out_path)

    layer_name = str(os.path.basename(os.path.splitext(out_path)[0]))
    layer = datasource.CreateLayer(layer_name, spat_ref, ogr.wkbPoint)

    # Construct a list of fields to add from the column list
    LOGGER.debug('Creating fields for the datasource')
    for field in columns:
        new_field = ogr.FieldDefn(field, ogr.OFTReal)
        layer.CreateField(new_field)

    LOGGER.debug('Entering iteration to create and set the features')
    # For each tuple key in row_order (for each point) create a point
    for row_point in row_order:
        point_dict = dict_data[row_point]

        geom_x = float(point_dict['x'])
        geom_y = float(point_dict['y'])

        geom = ogr.Geometry(ogr.wkbPoint)
        geom.AddPoint_2D(geom_x, geom_y)

        feature = ogr.Feature(layer.GetLayerDefn())
        layer.CreateFeature(feature)

        for field_name in columns:
            field_index = feature.GetFieldIndex(field_name)
            feature.SetField(field_index, point_dict[field_name])

        feature.SetGeometryDirectly(geom)
        layer.SetFeature(feature)

def rowcol_to_xy_uri(rows, cols, raster_uri):
    """converts row/col coordinates into x/y coordinates using raster_uri's
    geotransform

    Inputs:
        -rows: integer scalar or numpy array of row coordinates
        -cols: integer scalar or numpy array of column coordinates
        -raster_uri: uri from where the geotransform is going to be extracted

    Returns a tuple (X, Y) of scalars or numpy arrays of the projected
    coordinates"""

    raster = gdal.OpenEx(raster_uri)

    return rowcol_to_xy(row, cols, raster)

def rowcol_to_xy(rows, cols, raster):
    """non-uri version of rowcol_to_xy_uri"""
    gt = raster.GetGeoTransform()
    X = gt[0] + cols * gt[1] + rows * gt[2]
    Y = gt[3] + cols * gt[4] + rows * gt[5]

    return (X, Y)

def xy_to_rowcol_uri(x, y, raster_uri):
    """Does the opposite of rowcol_to_xy_uri"""
    raster = gdal.OpenEx(raster_uri)

    return xy_to_rowcol(x, y, raster)

def xy_to_rowcol(x, y, raster):
    """non-uri version of xy_to_rowcol_uri"""
    gt = raster.GetGeoTransform()
    igt = gdal.InvGeoTransform(gt)

    J = igt[0] + x * igt[1] + y * igt[2]
    I = igt[3] + x * igt[4] + y * igt[5]

    return (np.around(I).astype(int), np.around(J).astype(int))

def convert_tif_to_csv(tif_uri, csv_uri = None, mask = None):
    """ Converts a single band geo-tiff file to a csv text file

        Inputs:
            -tif_uri: the uri to the file to be converted
            -csv_uri: uri to the output file. The file should not exist.

        Outputs:
            -returns the ouput file uri

        returns nothing"""
    # The file has to exist
    message = 'File ' + tif_uri + ' does not exist.'
    assert os.path.isfile(tif_uri), message
    # The file has to be a tif
    extension = os.path.splitext(tif_uri)[1][1:].strip().lower()
    message = 'Wrong extension (' + extension + '), ' + \
        'expected ".tif" from File ' + tif_uri
    assert extension == 'tif', message
    # Create a uri for the new csv, and ensure it doesn't already exist
    if csv_uri == None:
        csv_uri = os.path.splitext(tif_uri)[0] + '.csv'
    #assert not os.path.exists(csv_uri), message

    dataset = gdal.OpenEx(tif_uri)
    assert dataset
    array = dataset.GetRasterBand(1).ReadAsArray()
    gt = dataset.GetGeoTransform()
    origin = np.array([gt[3], gt[0]])
    step = np.array([gt[5], gt[1]])
    size = np.array([dataset.RasterYSize, dataset.RasterXSize])
    # Add (step / 2.) to get the center of the pixels coordinates
    I = np.array(range(size[0])) * step[0] + origin[0] + (step[0] / 2.)
    J = np.array(range(size[1])) * step[1] + origin[1] + (step[1] / 2.)

    np.set_printoptions(precision = 20)
    with open(csv_uri, 'wb') as csv_file:
        writer = csv.writer(csv_file)

        file_basename = os.path.basename(os.path.splitext(csv_uri)[0])
        if file_basename[3] == '_':
            row_name = file_basename[4:]
        else:
            row_name = file_basename[2:]
        row = ['x', 'y', 'row', 'col', row_name]
        writer.writerow(row)
        if mask:
            for index in range(mask[0].size):
                i = mask[0][index]
                j = mask[1][index]
                row = [J[j], I[i], i, j, array[i, j]]
                writer.writerow(row)
        else:
            for i in range(size[0]):
                for j in range(size[1]):
                    row = [J[j], I[i], i, j, array[i, j]]
                    #row = [J[j], I[i], array[i, j]]
                    writer.writerow(row)
    return csv_uri

def convert_tifs_to_csv(tif_list, mask = None):
    csv_list = []
    for tif_uri in tif_list:
        csv_list.append(convert_tif_to_csv(tif_uri, mask = mask))
    return csv_list

def aggregate_csvs(csv_list, out_uri):
    """Concatenate 3-row csv files created with tif2csv

        Inputs:
            - csv_list: list of csv_uri strings

        Outputs:
            - uri_output: the output uri of the concatenated csv"""
    assert os.path.exists(csv_list[0])
    X = []
    Y = []
    I = []
    J = []
    headers = []
    with open(csv_list[0]) as csv_file:
        reader = csv.reader(csv_file)
        reader.next()
        expected_row_count = 0
        for row in reader:
            X.append(row[0])
            Y.append(row[1])
            I.append(row[2])
            J.append(row[3])
            expected_row_count += 1
        #print('found ' + str(expected_row_count) + ' rows in ' + csv_list[0])

    aggregated_data = []
    for csv_uri in csv_list:
        assert os.path.exists(csv_uri)
        # Make sure the number of rows is compatible (skip header)
        row_count = sum(1 for r in csv.reader(open(csv_uri))) - 1
        if row_count != expected_row_count:
            print(csv_uri + ' has ' + str(row_count) + ' rows, expected ' + \
            str(expected_row_count))
        data_column = []
        with open(csv_uri) as csv_file:
            reader = csv.reader(csv_file)
            header = reader.next()
            headers.append(header[4])
            for row in reader:
                data_column.append(row[4])
        aggregated_data.append(data_column)
    #print('header:', headers)

    # Sometimes, out_uri is the same name as an existing csv. To avoid
    # overriding errors and unwanted effects, the individual csvs will be
    # aggregated to a temporary file, and and will be copied to the final
    # location at the very end.
    temp_out_uri = natcap.invest.pygeoprocessing_0_3_3.geoprocessing.temporary_filename()
    np.set_printoptions(precision = 20)
    with open(temp_out_uri, 'wb') as csv_file:
        writer = csv.writer(csv_file)
        row = ['x', 'y', 'array row', 'array col']
        # Keep the CSV rows unsorted (in the order thefiles were concatenated)
        # We could simplify this code, but we might revert to a row
        # alphanumeric order, in which case we just have to comment this line
        # and uncomment the one just below.
        sorted_column = range(len(headers))
        #sorted_column = sorted(range(len(headers)), key = headers.__getitem__)
        # Sort CSV row names according to sorted_column and keep their sorted index
        # The list of column header names is alphabetically sorted
        for sc in sorted_column:
            row.append(headers[sc])
        writer.writerow(row)
        # The values below the header names are identically sorted
        for r in range(len(X)):
            row = [X[r], Y[r], I[r], J[r]]
            for sc in sorted_column:
                row.append(aggregated_data[sc][r])
            writer.writerow(row)
    shutil.copy(temp_out_uri, out_uri)
    return out_uri

def aggregate_tifs_from_list(uri_list, path, mask = None):
    if mask:
        assert os.path.isfile(mask), 'inexistant mask ' + mask
        raster = gdal.OpenEx(mask)
        assert raster, 'Invalid raster file ' + mask
        band = raster.GetRasterBand(1)
        nodata = band.GetNoDataValue()
        array = band.ReadAsArray()
        mask = np.where(array != nodata)

    tif_list = []
    for uri in uri_list:
        assert os.path.exists(uri)
        if '.tif' in uri:
            tif_list.append(uri)
    tif_count = len(tif_list)
    if not tif_count:
        print('No tifs found in', path)
        return
    dir_name = path.split(os.sep)[-1]
    out_uri = os.path.join(path, dir_name + '.csv')
    #print('aggregating ' + str(len(tif_list)) + ' tifs to ' + out_uri)
    #for f in tif_list:
    #    print(os.path.basename(f))
    csv_list = convert_tifs_to_csv(tif_list, mask = mask)
    #print('csv_list')
    #for csv in csv_list:
    #    print(os.path.basename(csv))
    aggregate_csvs(csv_list, out_uri)
    for obsolete_csv in csv_list:
        if obsolete_csv != out_uri:
            os.remove(obsolete_csv)

    return out_uri

def aggregate_tifs_from_directory(path = '.', mask = None):
    if mask:
        assert os.path.isfile(mask), 'inexistant mask ' + mask
        raster = gdal.OpenEx(mask)
        assert raster, 'Invalid raster file ' + mask
        band = raster.GetRasterBand(1)
        nodata = band.GetNoDataValue()
        array = band.ReadAsArray()
        mask = np.where(array != nodata)

    assert os.path.exists(path)
    base_dir = os.getcwd()
    os.chdir(path)

    uri_list = os.listdir('.')
    tif_list = []
    for uri in uri_list:
        # The file has to be a tif
        extension = os.path.splitext(uri)[1][1:].strip().lower()
#        print('uri', uri)
#        print('extension', extension)
        if extension in '.tif':
            tif_list.append(uri)
    # Sort the list of files alphabetically
    tif_list.sort()
    tif_count = len(tif_list)
    if not tif_count:
        print('No tifs found in', path)
        return
    dir_name = os.getcwd().split(os.sep)[-1]
    out_uri = os.path.join(os.getcwd(), dir_name + '.csv')
    #print('aggregating ' + str(len(tif_list)) + ' tifs from ' + \
    #    path + ' to ' + out_uri)
    csv_list = convert_tifs_to_csv(tif_list, mask = mask)
    aggregate_csvs(csv_list, out_uri)
    for obsolete_csv in csv_list:
        if obsolete_csv != out_uri:
            os.remove(obsolete_csv)

    os.chdir(base_dir)

    return out_uri

def save_local_wave_exposure_to_subdirectory(args):
    """ Copy local wave exposure to the outputs/ directory.

        Inputs:
            args['E_l']: uri to the local wave exposure data
            args['prefix']: prefix to be appended to the new filename

        Outputs:
            data_uri: dictionary containing the uri where the data is saved
    """
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    data_uri = {}
    data_uri['local_wave_exposure'] = \
        os.path.join(intermediate_directory, \
        args['prefix'] + 'local_wave_exposure.tif')
    shutil.copy(args['E_l'], data_uri['local_wave_exposure'])
    return data_uri

def save_oceanic_wave_exposure_to_subdirectory(args):
    """ Copy oceanic wave exposure to the outputs/ directory.

        Inputs:
            args['E_o']: uri to the oceanic wave exposure data
            args['prefix']: prefix to be appended to the new filename

        Outputs:
            data_uri: dictionary containing the uri where the data is saved
    """
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    data_uri = {}
    data_uri['oceanic_wave_exposure'] = \
        os.path.join(intermediate_directory, \
        args['prefix'] + 'oceanic_wave_exposure.tif')
    shutil.copy(args['E_o'], data_uri['oceanic_wave_exposure'])
    return data_uri

def save_wind_generated_waves_to_subdirectory(args):
    """ Copy the wave height and wave period to the outputs/ directory.

        Inputs:
            args['wave_height'][sector]: uri to "sector"'s wave height data
            args['wave_period'][sector]: uri to "sector"'s wave period data
            args['prefix']: prefix to be appended to the new filename

        Outputs:
            data_uri: dictionary containing the uri where the data is saved
    """
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])

    wave_height_list = args['wave_heights']
    wave_period_list = args['wave_periods']

    data_uri = {}
    for wave_height_uri in wave_height_list:
        shutil.copy(wave_height_uri, intermediate_directory)
    for wave_period_uri in wave_period_list:
        shutil.copy(wave_period_uri, intermediate_directory)
    return data_uri

def compute_continental_shelf_distance(args):
    """ Copy the continental shelf distance data to the outputs/ directory.

        Inputs:
            args['shore_shelf_distance']: uri to the continental shelf distance
            args['prefix']:

        Outputs:
            data_uri: a dictionary containing the uri where the data is saved.
    """
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    data_uri = {}
    data_uri['continental_shelf_distance'] = \
        os.path.join(intermediate_directory, \
        args['prefix'] + 'continental_shelf_distance.tif')
    shutil.copy(args['shore_shelf_distance'], \
        data_uri['continental_shelf_distance'])
    return data_uri

def compute_erodible_shoreline(args):
    """Compute the erodible shoreline as described in Greg's notes.
        The erodible shoreline is the shoreline segments of rank 5.

        Inputs:
            args[geomorphology]: the geomorphology data.
            args['prefix']: prefix to be added to the new filename.
            args['aoi_uri']: URI to the area of interest shapefile
            args['cell_size']: size of a cell on the raster

        Outputs:
            data_uri: a dictionary containing the uri where the data is saved.
    """
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    data_uri = {}
    data_uri['erodible_shoreline'] = os.path.join(intermediate_directory, \
    args['prefix'] + 'erodible_shoreline.tif')
    uri_list = [args['geomorphology']]
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets(uri_list, lambda x: 1. if x==5 else 0., \
    data_uri['erodible_shoreline'], gdal.GDT_Float32, 0., args['cell_size'], \
    'intersection', aoi_uri = args['aoi_uri'])
    return data_uri

def save_structure_to_subdirectory(args):
    """ Save structure data to its intermediate subdirectory, under a custom
    prefix.

        Inputs:
            args['structure_edges']: the data's uri to save to /outputs
            args['prefix']: prefix to add to the new filename. Currently used to
                mirror the labeling of outputs in Greg's notes.

        Outputs:
            data_uri: a dictionary of the uri where the data has been saved.
    """
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    data_uri = {}
    data_uri['structure_edges_subdir'] = \
        os.path.join(intermediate_directory, args['prefix'] + \
        'structure_edges.tif')
    shutil.copy(args['structure_edges'], data_uri['structure_edges_subdir'])
    return data_uri

def save_fetch_to_outputs(args):
    """ Function that copies the fetch information (depth and distances)
        in the outputs directory.

        Inputs:
            args['fetch_distance_uris']: A dictionary of ('string':string)
                entries where the first string is the sector in degrees, and
                the second string is a uri pointing to the file that contains
                the fetch distances for this sector.
            args['fetch_depths_uris']: A dictionary similar to the depth one,
                but the second string is pointing to the file that contains
                fetch depths, not distances.
            args['prefix']: String appended before the filenames. Currently
                used to follow Greg's output labelling scheme.

        Outputs:
            - data_uri that contains the uri of the new files in the outputs
                directory, one for fetch distance and one for fetch depths for
                each fetch direction 'n', for a total of 2n.
            """
    data_uri = {}
    prefix = args['prefix']
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    fetch_distance_uris = args['fetch_distance_uris']
    sectors_deg = \
    [os.path.splitext(fetch_distance_uris[s])[0].split('_')[-1] \
    for s in range(len(fetch_distance_uris))]
    for sector in range(len(fetch_distance_uris)):
        destination = os.path.join(intermediate_directory, \
            prefix + 'fetch_distance_' + \
            str(sectors_deg[sector]).zfill(3) + '.tif')
        shutil.copy(fetch_distance_uris[sector], destination)
        data_uri['fetch_distance_' + str(sectors_deg[sector]).zfill(3)] = \
            destination
    fetch_depth_uris = args['fetch_depth_uris']
    for sector in range(len(fetch_depth_uris)):
        destination = os.path.join(intermediate_directory, \
            prefix + 'fetch_depth_' + \
            str(sectors_deg[sector]).zfill(3) + '.tif')
        shutil.copy(fetch_depth_uris[sector], destination)
        data_uri['fetch_depth_' + str(sectors_deg[sector]).zfill(3)] = \
            destination
    return data_uri

def compute_erosion_exposure(args):
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    prefix = args['prefix'] + 'EROSION'
    data_uri = {}
    # Get the raw layer info from args
    info = args['outputs_info'][13]
    # Make sure this is what we expect
    message = 'Outputs info[13][0] does not refer to "1_l_", but to ' + \
    str(info)
    assert info[0] == '1_l_', message
    # List of layers used in the computation of the C.V.
    layer_uri_list = [args['shore_raster_uri']]
    # Extract the names of the available layers
    layer_names = info[3][0]
    for layer in layer_names:
        if layer in args:
            #LOGGER.info('adding layer ' + layer)
            layer_uri_list.append(args[layer])
    assert len(layer_uri_list) > 0
    # The function that computes the coastal vulnerability no habitats index:
    # 0s will be assigned to places other than the shoreline
    # The dividend should not count the shore, hence x.size - 1
    def erosion_exposure(*x):
        x = np.array(x)
        return np.power(np.prod(x), 1./(x.size-1))
    # Save new layer
    data_uri['erosion_no_structures'] = os.path.join(intermediate_directory, \
    prefix + '_without_structures.tif')
    data_uri['erosion_exposure'] = data_uri['erosion_no_structures']
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets(layer_uri_list, \
    erosion_exposure, data_uri['erosion_no_structures'], \
    gdal.GDT_Float32, 0., args['cell_size'], "intersection")

    # Adjust for structures if structure data exists
    if 'structure_edges' in args:
        # Input uris
        data_uri['erosion_exposure'] = \
        os.path.join(intermediate_directory, prefix + '_with_structures.tif')
        # Output uris
        uri_list = [data_uri['erosion_no_structures'], \
        args['structure_edges'], args['shore_raster_uri']]
        # Function that adjusts for structures
        def adjust_for_structures(CV, structure, shore):
            return max(1., CV - structure) * shore
        # Compute new layer
        natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets(uri_list, adjust_for_structures, \
        data_uri['erosion_exposure'], gdal.GDT_Float32, 0., \
        args['cell_size'], "intersection")
    return data_uri

def compute_habitat_role(args):
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])

    data_uri = {}

    def habitat_role(CV, CV_no_habitats, shore):
        if shore == 1:
            return (CV_no_habitats - CV)
        else:
            return 0.

    uri_list = [args['coastal_exposure'], \
    args['coastal_exposure_no_habitats'], args['shore_raster_uri']]
    data_uri['habitat_role'] = os.path.join(intermediate_directory, \
    args['prefix'] + 'habitat_role.tif')
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets(uri_list, habitat_role, \
    data_uri['habitat_role'], gdal.GDT_Float32, 0., args['cell_size'], \
    "intersection")

    return data_uri

def compute_coastal_exposure_no_habitats(args):
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    prefix = args['prefix'] + 'CV_no_hab'
    data_uri = {}

    layer_uri_list = []  # List of layers used in the computation of the C.V.
    # Get the raw layer info from args
    info = args['outputs_info'][10]
    # Make sure this is what we expect
    message = 'Outputs info[10] does not refer to "1_i_", but to ' +str(info[0])
    assert info[0] == '1_i_', message
    # Extract the names of the available layers
    layer_names = \
    [layer[0] for layer in info[3] if layer[0] is not 'natural_habitats']
    for layer in layer_names:
        if layer in args:
            #LOGGER.info('adding layer ' + layer)
            layer_uri_list.append(args[layer])
    assert len(layer_uri_list) > 0
    layer_uri_list.append(args['shore_raster_uri'])
    # The function that computes the coastal vulnerability no habitats index:
    # Add 5 to the list of parameters, to account for natural_habitats == 5
    # The dividend should not count the shore, but add hab: x.size + 1 - 1
    # 0s will be assigned to places other than the shoreline
    def CV_no_habitats_index(*x):
        x = x + (5,)
        x = np.array(x)
        return np.power(np.prod(x), 1./(x.size - 1))
    # Save new layer
    data_uri['coastal_exposure_no_habitats'] = \
    os.path.join(intermediate_directory, \
    args['prefix'] + 'coastal_exposure_no_habitats.tif')
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets(layer_uri_list, \
    CV_no_habitats_index, data_uri['coastal_exposure_no_habitats'], \
    gdal.GDT_Float32, 0., args['cell_size'], "intersection")

    return data_uri

def compute_coastal_exposure(args):
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    prefix = args['prefix'] + 'CV'
    data_uri = {}
    # Get the raw layer info from args
    info = args['outputs_info'][10]
    # Make sure this is what we expect
    message = 'Outputs info[10] does not refer to "1_i_", but to ' +str(info[0])
    assert info[0] == '1_i_', message
    # List of layers used in the computation of the C.V.
    layer_uri_list = [args['shore_raster_uri']]
    # Extract the names of the available layers
    layer_names = [layer[0] for layer in info[3]]
    for layer in layer_names:
        if layer in args:
            LOGGER.info('adding layer ' + layer)
            layer_uri_list.append(args[layer])
    assert len(layer_uri_list) > 0
    # The function that computes the coastal vulnerability index:
    # 0s will be assigned to places other than the shoreline
    # The dividend should not count the shore, hence x.size - 1
    def coastal_exposure_index(*x):
        x = np.array(x)
        return np.power(np.prod(x), 1./(x.size - 1))
    # Save new layer
    data_uri['coastal_exposure'] = os.path.join(intermediate_directory, \
    args['prefix'] + 'coastal_exposure.tif')
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets(layer_uri_list, \
    coastal_exposure_index, data_uri['coastal_exposure'], \
    gdal.GDT_Float32, 0., args['cell_size'], "intersection")

    return data_uri

def compute_segment_exposure(args):
    """ Compute exposed and sheltered shoreline segment map."""
    no_data = -1.0
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    data_uri = {}
    fetch_distances = args['fetch_distances']
    fetch_depths = args['fetch_depths']
    max_fetch = args['max_fetch']
    depth_threshold = args['depth_threshold']

    shore_raster = gdal.OpenEx(args['shore_raster_uri'])
    shore_array = shore_raster.GetRasterBand(1).ReadAsArray()
    # Number (count) of enclosed/shallow rays
    depth_count = np.ones_like(shore_array) * no_data
    distance_count = np.ones_like(shore_array) * no_data
    # Average depth
    avg_depth = np.ones_like(shore_array) * no_data
    avg_distance = np.ones_like(shore_array) * no_data
    # Percentage of enclosed/shallow rays
    depth_pct = np.ones_like(shore_array) * no_data
    distance_pct = np.ones_like(shore_array) * no_data
    # Boolean value of enclosed/shallow criteria for each segment
    deep_segments = np.ones_like(shore_array) * no_data
    open_segments = np.ones_like(shore_array) * no_data

    # Compute the exposure threshold and round it to the closest integer
    exposure_proportion = args['exposure_proportion']

    # Determine if each shoreline segment is open and deep
    # and assign the segment 3 if sheltered, or 4 if not:
    shore_keys = fetch_distances.keys()
    shore_segments = zip(*shore_keys)
    shore_segments = (np.array(shore_segments[0]), np.array(shore_segments[1]))
    I, J = xy_to_rowcol(shore_segments[0], shore_segments[1], shore_raster)
    shore_segments = zip(I, J)
    shore_exposure = np.ones_like(shore_array) * no_data
    # Each segment has a numpy array for the fetch distances and a numpy array
    # for the fetch depth
    I = []  # Not all the values in I and J are valid. Create valid I and J
    J = []  # in the loop
    np.set_printoptions(precision = 1)
    for s in range(len(shore_segments)):
        segment = shore_segments[s]
        key = shore_keys[s]
        # The segments might fall outside of distance_count because
        # they come from a raster file that might be bigger than distance_count
        if ((segment[0] < 0) or \
            (segment[0] >= distance_count.shape[0]) or \
            (segment[1] < 0) or \
            (segment[1] >= distance_count.shape[1])):
            continue
        # This is a valid segment, add it to I and J
        I.append(segment[0])
        J.append(segment[1])
        distances = fetch_distances[key]
        depths = fetch_depths[key] * -1 # make depths positive
        # Remove rays that reach land on the first pixel:
        valid_rays = \
            np.where((distances >= args['cell_size']) & (depths >= 0.))
        valid_rays_count = valid_rays[0].size
        count_threshold = int(round(exposure_proportion * valid_rays_count))

        distances = distances[valid_rays]
        depths = depths[valid_rays]

        # Rays not blocked by land
        long_rays = np.sum((distances >= max_fetch).astype(int))
        # Rays profile deeper than threshold on average
        deep_rays = np.sum((depths >= depth_threshold).astype(int))

        # Adjust for sheltered segment and avoid division by zero
        if valid_rays_count == 0:
            # No valid rays: segment is over-sheltered
            is_segment_open = False # Segment is surrounded by land
            is_segment_deep = False # Segment is expected to be shallow

            distance_count[segment] = 0
            avg_distance[segment] = no_data
            distance_pct[segment] = 0.
            open_segments[segment] = 0
            depth_count[segment] = 0
            avg_depth[segment] = no_data
            depth_pct[segment] = 0.
            deep_segments[segment] = 0

        # Don't know exposure: compute it
        else:
            is_segment_open = long_rays >= count_threshold
            is_segment_deep = deep_rays >= count_threshold

            distance_count[segment] = long_rays
            avg_distance[segment] = np.average(distances)
            distance_pct[segment] = \
                float(long_rays) / valid_rays_count
            open_segments[segment] = is_segment_open
            depth_count[segment] = deep_rays
            avg_depth[segment] = np.average(depths)
            depth_pct[segment] = \
                float(deep_rays) / valid_rays_count
            deep_segments[segment] = is_segment_deep

        # If too few valid rays, the segment should probably
        # be considered sheltered
        if valid_rays_count <= 3*args['rays_per_sector']:
            is_segment_open = False # Segment is too landlocked


        # segment is exposed
        if is_segment_open and is_segment_deep:
            shore_exposure[segment] = EXPOSED_SHORE
        # segment is not exposed
        else:
            shore_exposure[segment] = SHELTERED_SHORE


    # We know the list of valid segments, convert them to numpy arrays
    I = np.array(I).astype(int)
    J = np.array(J).astype(int)

#    distance_count[(I, J)] = 1
    data_uri['long_rays'] = \
    save_array_to_raster(distance_count, \
    os.path.join(intermediate_directory, \
    args['prefix'] + '1_long_fetch_rays.tif'), args['shore_raster_uri'], \
    args['cell_size'], no_data = no_data)

    data_uri['fetch_distance_average'] = \
    save_array_to_raster(avg_distance, \
    os.path.join(intermediate_directory, \
    args['prefix'] + '2_fetch_distances_average.tif'), \
    args['aoi_raster_uri'], args['cell_size'], no_data = no_data)

    data_uri['open_segments_pct'] = \
    save_array_to_raster(distance_pct, \
    os.path.join(intermediate_directory, \
    args['prefix'] + '3_open_segments_pct.tif'), \
    args['aoi_raster_uri'], args['cell_size'], no_data = no_data)

    data_uri['open_segments_bool'] = \
    save_array_to_raster(open_segments, \
    os.path.join(intermediate_directory, \
    args['prefix'] + '4_open_segments_bool.tif'), \
    args['aoi_raster_uri'], args['cell_size'], no_data = no_data)

    data_uri['deep_segments'] = \
    save_array_to_raster(depth_count, \
    os.path.join(intermediate_directory, \
    args['prefix'] + '5_deep_segments.tif'), \
    args['aoi_raster_uri'], args['cell_size'], no_data = no_data)

    data_uri['fetch_depths_average'] = \
    save_array_to_raster(avg_depth, \
    os.path.join(intermediate_directory, \
    args['prefix'] + '6_fetch_depths_average.tif'), \
    args['aoi_raster_uri'], args['cell_size'], no_data = no_data)

    data_uri['deep_segments_pct'] = \
    save_array_to_raster(depth_pct, \
    os.path.join(intermediate_directory, \
    args['prefix'] + '7_deep_segments_pct.tif'), \
    args['aoi_raster_uri'], args['cell_size'], no_data = no_data)

    data_uri['deep_segments_bool'] = \
    save_array_to_raster(deep_segments, \
    os.path.join(intermediate_directory,\
    args['prefix'] + '8_deep_segments_bool.tif'), \
    args['aoi_raster_uri'], args['cell_size'], no_data = no_data)

    data_uri['shore_exposure'] = \
    save_array_to_raster(shore_exposure, \
    os.path.join(intermediate_directory, \
    args['prefix'] + 'shore_exposure.tif'), \
    args['aoi_raster_uri'], args['cell_size'], no_data = no_data)
    # Initialize the shoreline to 1s
    shore_array[shore_exposure != no_data] = 1
    # Mask the shoreline for which we don't want an output
    if args['area_computed'] == 'sheltered':
        shore_array[shore_exposure == EXPOSED_SHORE] = 0
    # Save the new shore over which all the computation will now take place
    shore_raster = gdal.OpenEx(args['shore_raster_uri'], gdal.GA_Update)
    shore_raster.GetRasterBand(1).WriteArray(shore_array)
    shore_raster = None

    return data_uri

def compute_structure_protection(args):
    """Compute the structure influence on the shore to later include it in the
    computation of the layers final rankings, as is specified in Gregg's
    the additional notes (decrement ranks around structure edges).

        Inputs:
            - args['aoi_uri']: string uri to the datasource of the area of
                interest
            - args['shore_raster_uri']: dataset uri of the coastline within the AOI
            - args['structures_uri']: string of the structure datasource uri
            - args['cell_size']: integer of the size of a pixel in meters
            - args['intermediate_directory']: string of the uri where
                intermediate files are stored
            - args['prefix']: string prefix appended to every intermediate file

        Outputs:
            - data_uri: a dictionary of the file uris generated in the
              intermediate directory.
            - data_uri['adjusted_structures']: string of the dataset uri
              obtained from reprojecting args['structures_uri'] and burining it
              onto the aoi. Contains the structure information across the whole
              aoi.
            - data_uri['shore_structures']: string uri pointing to the
              structure information along the coast only.
            - data_uri['structure_influence']: string uri pointing to a
              datasource of the spatial influence of the structures.
            - data_uri['structure_edge']: string uri pointing to the datasource
              of the edges of the structures.
            """
    cell_size = args['cell_size']
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    prefix = args['prefix'] + 'STRUCT'
    file_count = 0
    # Dictionary that contains the data's uri indexed by name
    data_uri = {}
    # Create a kernel to detect the structure edges:
    #   - values >= 9: the shore is armored
    #   - values between 1 and 8: more vulnerable shore segment
    #   - values == 0: segment not influenced by a structure
    kernel = np.array([[-1, -1, -1],
                       [-1, -9, -1],
                       [-1, -1, -1]])
    # Compute the structure edge using convolution:
    shore_structures = gdal.OpenEx(args['structures_raster_uri'])
    shore_structures_array = shore_structures.GetRasterBand(1).ReadAsArray()
    structure_influence = sp.signal.convolve2d(shore_structures_array, \
        kernel, mode='same').astype('int')
    data_uri['structure_influence'] = \
        save_array_to_raster(structure_influence, \
        os.path.join(intermediate_directory, \
        prefix + '_' + str(file_count) + '_structure_influence.tif'), \
        args['aoi_raster_uri'], args['cell_size'], no_data = 0.)
    file_count += 1
    # Adjust shore classification from the convolution:
    #   - near structure shore segment: 1 (convolved values -1 to -8)
    #   - off structure shore segment: 0 (convolved value of 0)
    data_uri['structure_edges'] = os.path.join(intermediate_directory, \
        args['prefix'] + 'structure_edges.tif')

    def compute_structure_edge(influence, shore):
        # Not on the shore: return nodata
        if shore == 0:
            return 0
        # On the shore, investigate further
        else:
            # On the structure edge: return 1
            if (influence < 0) and (influence > -9):
                return 1
            # Not on the structure edge: return nodata
            else:
                return 0

    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets( \
    [data_uri['structure_influence'], args['shore_raster_uri']], \
    compute_structure_edge, data_uri['structure_edges'], \
    gdal.GDT_Byte, 0, cell_size, 'union', aoi_uri = args['aoi_uri'])

    return data_uri

def compute_coastal_population(args):
    """Compute population living along the shore within a given radius.

        Inputs:
            - args['intermediate_directory']: uri to a directory where
              intermediate files are stored
            - args['subdirectory']: string URI of an existing subdirectory
            - args['prefix']: string prefix appended to every file generated
            - args['population_uri']: uri to the population density dataset.
            - args['population_radius']: used to compute the population density.
            - args['aoi_uri']: uri to a polygon shapefile
            - args['cell_size']: size of a pixel in meters

        Outputs:
            - Return a uri dictionary of all the files created to generate the
              population density along the coastline."""
    cell_size = args['cell_size']
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    prefix = args['prefix'] + 'COASTAL_POP'
    file_count = 0
    population_radius = args['population_radius']
    # Dictionary that contains the data's uri indexed by name
    data_uri = {}
    # Create a kernel to gather the total population within a given radius
    R = (population_radius/cell_size) if population_radius >= cell_size else 1.
    kernel = disc_kernel(R)
    # Convolve the population with the kernel
    raster = gdal.OpenEx(args['population_raster_uri'])
    band = raster.GetRasterBand(1)
    pop_array = band.ReadAsArray()
    nodata = band.GetNoDataValue()
    band = None
    raster = None
    pop_array[pop_array == nodata] = 0
    pop_density = sp.signal.convolve2d(pop_array, kernel, mode='same')
    # Save population density to a raster
    data_uri['pop_density'] = os.path.join(intermediate_directory, \
        prefix + '_' + str(file_count) + "_pop_density.tif")
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.new_raster_from_base_uri(args['population_raster_uri'], \
        data_uri['pop_density'], 'GTiff', 0., gdal.GDT_Float32)
    target_raster = gdal.OpenEx(data_uri['pop_density'], gdal.GA_Update)
    target_raster.GetRasterBand(1).WriteArray(pop_density)
    target_raster = None
    file_count += 1
    # Compute the population along the coast
    data_uri['coastal_population'] = \
        os.path.join(intermediate_directory, \
        args['prefix'] + 'coastal_population.tif')
    file_count += 1
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets( \
        [data_uri['pop_density'], args['shore_raster_uri']], \
        lambda x,y: x*y, data_uri['coastal_population'], \
        gdal.GDT_Float32, 0., args['cell_size'], \
        'union', aoi_uri = args['aoi_uri'])
    # Done
    return data_uri

def compute_surge_potential(args):
    """Compute surge potential index as described in the user manual.

        Inputs:
            - args['bathymetry']: bathymetry DEM file.
            - args['landmass']: shapefile containing land coverage data (land =
              1, sea = 0)
            - args['aoi_uri']: uri to the datasource of the area of interest
            - args['shore_raster_uri']: uri to a shore raster where the shoreline is 1,
              and everything else is 0.
            - args['cell_size']: integer number for the cell size in meters
            - args['intermediate_directory']: uri to the directory where
              intermediate files are stored

        Output:
            - Return R_surge as described in the user guide.

        Intermediate outputs:
            - rasterized_sea_level_rise.tif:rasterized version of the shapefile
            - shore_level_rise.tif: sea level rise along the shore.
            - sea_level_rise.tif: sea level rise index along the shore."""
    exposure_raster = gdal.OpenEx(args['shore_exposure'])
    exposure_array = exposure_raster.GetRasterBand(1).ReadAsArray()
    exposed_segments = np.where(exposure_array == 1)
    shore_points = np.where(exposure_array >= 0)
    exposure_array = None
    exposure_raster = None
    cell_size = args['cell_size']
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    prefix = args['prefix'] + 'SURGE'
    file_count = 0
    nodata = -1
    # Dictionary that contains the data's uri indexed by name
    data_uri = {}

    # Get the edge of the continental shelf
    base_raster_uri = None
    if 'continental_shelf_raster_uri' in args:
        continental_shelf_raster = \
            gdal.OpenEx(args['continental_shelf_raster_uri'])
        continental_shelf = \
            continental_shelf_raster.GetRasterBand(1).ReadAsArray()
        base_raster_uri = args['continental_shelf_raster_uri']
    else:
        depth_contour = args['depth_contour']
        bathymetry_raster = gdal.OpenEx(args['bathymetry_raster_uri'])
        continental_shelf = bathymetry_raster.GetRasterBand(1).ReadAsArray()
        continental_shelf[continental_shelf >= -depth_contour] = 1 #on the shelf
        continental_shelf[continental_shelf < -depth_contour] = 0 #off the shelf
        base_raster_uri = args['bathymetry_raster_uri']
    data_uri['continental_shelf'] = os.path.join(intermediate_directory, \
        prefix + '_' + str(file_count) + '_continental_shelf.tif')
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.new_raster_from_base_uri(base_raster_uri,
    data_uri['continental_shelf'], 'GTiff', nodata, gdal.GDT_Float32)
    raster = gdal.OpenEx(data_uri['continental_shelf'], gdal.GA_Update)
    raster.GetRasterBand(1).WriteArray(continental_shelf)
    raster = None
    file_count += 1
    # Compute the distances on the shelf from the edge
    distance_to_shelf = morphology.distance_transform_edt(continental_shelf)
    data_uri['distance_to_shelf'] = os.path.join(intermediate_directory, \
        prefix + '_' + str(file_count) + '_distance_to_shelf.tif')
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.new_raster_from_base_uri(base_raster_uri,
    data_uri['distance_to_shelf'], 'GTiff', nodata, gdal.GDT_Float32)
    raster = gdal.OpenEx(data_uri['distance_to_shelf'], gdal.GA_Update)
    raster.GetRasterBand(1).WriteArray(distance_to_shelf)
    raster = None
    file_count += 1
    # Compute the surge along the shore in meters
    shore_shelf_distance = distance_to_shelf * cell_size
    data_uri['shore_shelf_distance'] = os.path.join(intermediate_directory, \
        prefix + '_' + str(file_count) + '_shore_shelf_distance.tif')
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.new_raster_from_base_uri(base_raster_uri,
    data_uri['shore_shelf_distance'], 'GTiff', nodata, gdal.GDT_Float32)
    raster = gdal.OpenEx(data_uri['shore_shelf_distance'], gdal.GA_Update)
    raster.GetRasterBand(1).WriteArray(shore_shelf_distance)
    raster = None
    file_count += 1
    # Use percentiles to assign vulnerability index
    R_surge = np.ones_like(shore_shelf_distance) * nodata
    R_surge[shore_points] = 0
    R_surge[exposed_segments] = \
        rank_shore(shore_shelf_distance[exposed_segments], 5) + 1
    # Save the exposed shore segments ranks
    data_uri['exposed_surge_ranks'] = os.path.join(intermediate_directory, \
        prefix + '_' + str(file_count) + '_exposed_surge_ranks.tif')
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.new_raster_from_base_uri(base_raster_uri,
    data_uri['exposed_surge_ranks'], 'GTiff', nodata, gdal.GDT_Float32)
    raster = gdal.OpenEx(data_uri['exposed_surge_ranks'], gdal.GA_Update)
    raster.GetRasterBand(1).WriteArray(R_surge)
    raster = None
    # Assign closest exposed shore segments to sheltered ones
    data_uri['surge_potential'] = \
        os.path.join(intermediate_directory, args['prefix'] + \
            'surge_potential.tif')
    assign_sheltered_segments(args['shore_exposure'],
        data_uri['exposed_surge_ranks'], data_uri['surge_potential'])
    # done
    return data_uri

def compute_additional_layer(args):
    """Compute the additional layer the sea level rise index.

        Inputs:
            -args['additional_layer_uri']: uri to the additional layer data.
            -args['aoi_uri']: uri to datasource of the area of interest
            -args['shore_raster_uri']: uri to the shoreline dataset (land =1, sea =0)
            -args['cell_size']: integer of the cell size in meters
            -args['intermediate_directory']: uri to the intermediate file
             directory

        Output:
            - Return a dictionary of all the intermediate file URIs.

        Intermediate outputs:
            - rasterized_sea_level_rise.tif:rasterized version of the shapefile
            - shore_FIELD_NAME.tif: raw value along the shore.
            - FIELD_NAME.tif: index along the shore. If all
                the shore has the same value, assign the moderate index value 3.
            """
    shore_raster = gdal.OpenEx(args['shore_raster_uri'])
    shore_array = shore_raster.GetRasterBand(1).ReadAsArray()
    shore_raster = None
    shore_points = np.where(shore_array == 1)
    cell_size = args['cell_size']
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    prefix = args['prefix']
    nodata = -1
    file_count = 0
    # Dictionary that contains the data's uri indexed by name
    data_uri = {}
    # Extract the array from the shapefile
    sea_rise_raster = gdal.OpenEx(args['additional_layer_raster_uri'])
    sea_rise_array = sea_rise_raster.GetRasterBand(1).ReadAsArray()
    # Compute sea level rise along the shore
    # TODO: replace by vectorize_datasets
    R_slr = np.ones_like(sea_rise_array) * nodata
    R_slr[shore_points] = sea_rise_array[shore_points]
    data_uri['additional_layer_raw'] = \
        save_array_to_raster(R_slr, \
        os.path.join(intermediate_directory, \
        prefix + str(file_count) + '_additional_layer_raw.tif'), \
        args['aoi_raster_uri'], cell_size, no_data = nodata)
    file_count += 1
    # Convert the rise to a rank
    ranks = rank_shore(R_slr[shore_points], 5) + 1
    R_slr[shore_points] = ranks
    data_uri['additional_layer'] = \
        save_array_to_raster(R_slr, \
        os.path.join(intermediate_directory, \
        prefix + str(file_count) + '_additional_layer_ranked.tif'), \
        args['aoi_raster_uri'], cell_size, no_data = nodata)
    # Done
    return data_uri

# Update the docstring!
def compute_sea_level_rise(args):
    """Compute the sea level rise index as described in the user manual.

        Inputs:
            -args['sea_level_rise']: shapefile with the sea level rise data.
            -args['aoi_uri']: uri to datasource of the area of interest
            -args['shore_raster_uri']: uri to the shoreline dataset (land =1, sea =0)
            -args['cell_size']: integer of the cell size in meters
            -args['intermediate_directory']: uri to the intermediate file
             directory

        Output:
            - Return a dictionary of all the intermediate file URIs.

        Intermediate outputs:
            - rasterized_sea_level_rise.tif:rasterized version of the shapefile
            - shore_level_rise.tif: sea level rise along the shore.
            - sea_level_rise.tif: sea level rise index along the shore. If all
                the shore has the same value, assign the moderate index value 3.
            """
    shore_raster = gdal.OpenEx(args['shore_raster_uri'])
    shore_array = shore_raster.GetRasterBand(1).ReadAsArray()
    shore_raster = None
    shore_points = np.where(shore_array == 1)
    exposure_raster = gdal.OpenEx(args['shore_exposure'])
    exposure_array = exposure_raster.GetRasterBand(1).ReadAsArray()
    exposed_segments = np.where(exposure_array == 1)
    sheltered_segments = np.where(exposure_array == 0)
    exposure_array = None
    exposure_raster = None
    cell_size = args['cell_size']
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    prefix = args['prefix'] + 'SLR'
    nodata = 0.
    file_count = 0
    # Dictionary that contains the data's uri indexed by name
    data_uri = {}
    # Extract the array from the shapefile
    sea_rise_raster = gdal.OpenEx(args['sea_level_rise_raster_uri'])
    sea_rise_array = sea_rise_raster.GetRasterBand(1).ReadAsArray()
    # Compute sea level rise along the shore
    # TODO: replace by vectorize_rasters
    R_slr = np.ones_like(sea_rise_array) * nodata
    R_slr[shore_points] = sea_rise_array[shore_points]
    data_uri['shore_level_rise'] = \
        save_array_to_raster(R_slr, \
        os.path.join(intermediate_directory, \
        prefix + '_' + str(file_count) + '_sea_level_rise.tif'), \
        args['aoi_raster_uri'], cell_size, no_data = nodata)
    file_count += 1
    # Convert the rise to a rank
    R_slr[exposed_segments] = rank_shore(R_slr[exposed_segments], 5) + 1
    R_slr[sheltered_segments] = nodata
    data_uri['sea_level_rise_ranks'] = \
        save_array_to_raster(R_slr, \
        os.path.join(intermediate_directory, \
        prefix + '_' + str(file_count) + \
        '_exposed_sea_level_rise_ranks.tif'), \
        args['aoi_raster_uri'], cell_size, no_data = nodata)
    # Assign closest exposed shore segments to sheltered ones
    data_uri['sea_level_rise'] = \
        save_array_to_raster(R_slr, \
        os.path.join(intermediate_directory, \
        args['prefix'] + 'sea_level_rise.tif'), \
        args['aoi_raster_uri'], cell_size, no_data = nodata)
    #print("args['shore_exposure']", args['shore_exposure'])
    assign_sheltered_segments(args['shore_exposure'],
        data_uri['sea_level_rise_ranks'], data_uri['sea_level_rise'])
    # Done
    return data_uri

def compute_natural_habitats_vulnerability(args):
    """Compute the natural habitat rank as described in the user manual.

        Inputs:
            -args['habitats_csv_uri']: uri to a comma-separated text file
             containing the list of habitats.
            -args['habitats_directory_uri']: uri to the directory where to find
             the habitat shapefiles.
            -args['aoi_uri']: uri to the datasource of the area of interest
            -args['shore_raster_uri']: uri to the shoreline dataset
                (land =1, sea =0)
            -args['cell_size']: integer cell size in meters
            -args['intermediate_directory']: uri to the directory where
              intermediate files are stored

        Output:
            -data_uri: a dictionary of all the intermediate file URIs.

        Intermediate outputs:
            - For each habitat (habitat name 'ABCD', with id 'X') shapefile:
                - ABCD_X_raster.tif: rasterized shapefile data.
                - ABCD_influence.tif: habitat area of influence. Convolution
                  between the rasterized shape data and a circular kernel which
                    radius is the habitat's area of influence, TRUNCATED TO
                    CELL_SIZE!!!
                - ABCD_influence_on_shore.tif: habitat influence along the shore
            - habitats_available_data.tif: combined habitat rank along the
                shore using equation 4.4 in the user guide.
            - habitats_missing_data.tif: shore section without habitat data.
            - habitats.tif: shore ranking using habitat and default ranks."""
    shore = gdal.OpenEx(args['shore_raster_uri'])
    shore_array = shore.GetRasterBand(1).ReadAsArray()
    shore_array_shape = shore_array.shape
    shore_points = np.where(shore_array == 1)
    shore_array = None
    shore = None
    cell_size = args['cell_size']
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    prefix = args['prefix'] + 'HAB_'
    # Predefined constants
    NAME                = 0 # Habitat name (string)
    RANK                = 2 # Habitat rank (int, 1 to 5)
    PROTECTION_DISTANCE = 3 # Habitat protection distance in meters (int)
    RASTER_URI          = 5 # Raster URI
    # Dictionary that contains the data's uri indexed by name
    data_uri = {}

    R = {}  # Dictionary: key=segment coord., value=list of fronting habitats
    prefix_number = 1   # used to keep track of file creation order
    # Process each habitat

    shipped_lines = 0
    for habitat_data in args['habitat_data'].values():
        # Breaks if the habitat is empty (no RASTER_URI stored)
        message = "The model couldn't find a raster file for habitat '" + \
            habitat_data[NAME] + "'. Please make sure the habitat " + \
            "and the corresponding file are spelled correctly."
        assert len(habitat_data) > RASTER_URI, message

        # See if we should skip this habitat:
        if habitat_data[RASTER_URI] == 'skip':
            continue

        # A few shorthand names, for convenience
        habitat_name = habitat_data[NAME]
        habitat_file_name = habitat_name.replace(' ', '_')
        habitat_rank = int(habitat_data[RANK])
        habitat_distance = int(habitat_data[PROTECTION_DISTANCE])
        habitat_uri = habitat_data[RASTER_URI]
        habitat_raster = gdal.OpenEx(habitat_uri)
        message = "The habitat raster '" + str(habitat_uri) + \
            "' cannot be opened. Please make sure the file exists " + \
            "or that habitat " + habitat_name + " in " + \
            args['habitats_csv_uri'] + " is spelled correctly."
        assert habitat_raster is not None, message
        habitat_band = habitat_raster.GetRasterBand(1)
        habitat_array = habitat_band.ReadAsArray()
        prefix_number += 1
        # Create a kernel to detect where the habitat influences the shore
        kernel = disc_kernel(habitat_distance / cell_size)
        # Area of influence by convolving the kernel with the habitat raster
        influenced_area = \
            sp.signal.convolve2d(habitat_array, kernel, mode='same')
        data_uri[habitat_file_name + '_influence'] = \
        save_array_to_raster(influenced_area, \
            os.path.join(intermediate_directory, \
            prefix + str(prefix_number).zfill(2) + '_' + \
            habitat_file_name +'_influence.tif'), \
            args['aoi_raster_uri'], cell_size, no_data = 0.)
        prefix_number += 1
        # Compute the influence of the habitat along the shore
        # TODO: replace by vectorize_rasters
        shore_influence = np.zeros_like(influenced_area)
        shore_influence[shore_points] = influenced_area[shore_points]
        data_uri[habitat_file_name + '_influence_on_shore'] = \
        save_array_to_raster(shore_influence, \
            os.path.join(intermediate_directory, \
            prefix + str(prefix_number).zfill(2) + '_' + \
            habitat_file_name + '_influence_on_shore.tif'), \
            args['aoi_raster_uri'], cell_size, no_data = 0.)
        prefix_number += 1
        # Mark all the shore segments under the influence of this habitat
        [I, J] = np.where(shore_influence > 0)
        for point in zip(I, J):
            if point in R:
                R[point] = \
                np.concatenate([R[point], np.array([habitat_rank])])
            else:
                R[point] = np.array([habitat_rank])

    # TODO: replace by vectorize_rasters
    shore_data = np.ones(shore_array_shape) * -1 # Will hold known ranking data
    shore_data[shore_points] = 0
    # For each point influenced by a natural habitat
    for point in R:
        # Compute the vulnerability index
        R[point] = combined_rank(R[point])
        # Fill the array with shore data
        shore_data[point[0], point[1]] = R[point]

    data_uri['habitats_available_data'] = \
        os.path.join(intermediate_directory, \
        prefix + str(prefix_number).zfill(2) + '_' + \
        'habitats_available_data.tif')
    save_array_to_raster(shore_data, data_uri['habitats_available_data'], \
        args['aoi_raster_uri'], cell_size, no_data = 0.0)
    prefix_number += 1
    # Default shore vulnerability index
    # TODO: replace by vectorize_rasters
    default_data = np.ones_like(shore_data) * -1
    default_data[shore_points] = 0.
    default_data[np.where(shore_data == 0)] = 5
    data_uri['habitats_missing_data'] = \
        os.path.join(intermediate_directory, \
        prefix + str(prefix_number).zfill(2) + '_' + \
        'habitats_missing_data.tif')
    save_array_to_raster(default_data, data_uri['habitats_missing_data'], \
        args['aoi_raster_uri'], cell_size, no_data = -1.0)
    # Actual shore vulnerability index
    # TODO: replace by vectorize_rasters
    R_hab = np.ones_like(shore_data) * 0.
    R_hab[shore_points] = \
        shore_data[shore_points] + default_data[shore_points]
    data_uri['natural_habitats'] = \
    os.path.join(intermediate_directory, \
    args['prefix'] + 'natural_habitats.tif')
    save_array_to_raster(R_hab, data_uri['natural_habitats'], \
    args['aoi_raster_uri'], cell_size, no_data = 0.)
    return data_uri

# TODO: update docstring
def compute_geomorphology(args):
    """Translate geomorphology RANKS to shore pixels.

    Create a raster identical to the shore pixel raster that has geomorphology
    RANK values. The values are gathered by finding the closest geomorphology
    feature to the center of the pixel cell.

    Parameters:
        args['geomorphology_uri'] (string): a path on disk to a shapefile of
            the gemorphology ranking along the coastline.
        args['shore_raster_uri'] (string): a path on disk to a the shoreline
            dataset (land = 1, sea = 0).
        args['intermediate_directory'] (string): a path to the directory where
            intermediate files are stored.
        args['subdirectory'] (string): a path for a directory to store the
            specific geomorphology intermediate steps.

    Returns:
        data_uri (dict): a dictionary of with the path for the geomorphology
            raster.
    """

    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    prefix = args['prefix'] + 'GEO_'

    # Dictionary that contains the data's uri indexed by name
    data_uri = {}

    points_from_shore_path = os.path.join(
        intermediate_directory, 'points_from_shore.shp')

    # Create point shapefile from shore raster
    raster_to_point_vector(args['shore_raster_uri'], points_from_shore_path)

    # Find the closest geomorphology polyline to each point and push
    # the geomorphology rank onto the point
    inherit_field = 'RANK'
    nearest_vector_neighbor(
        args['geomorphology_uri'], points_from_shore_path, inherit_field)

    geomorphology_raster_path = os.path.join(
        intermediate_directory, args['prefix'] + "geomorphology.tif")

    nodata = natcap.invest.pygeoprocessing_0_3_3.get_nodata_from_uri(args['shore_raster_uri'])
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.new_raster_from_base_uri(
        args['shore_raster_uri'], geomorphology_raster_path, 'GTiff', nodata,
        gdal.GDT_Int32)

    # Burn the ranks from the point shapefile back onto a raster
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.rasterize_layer_uri(
        geomorphology_raster_path, points_from_shore_path,
        option_list=["ATTRIBUTE=RANK", "ALL_TOUCHED=TRUE"])

    data_uri['geomorphology'] = geomorphology_raster_path
    return data_uri

def raster_to_point_vector(raster_path, point_vector_path):
    """Create a point shapefile from raster pixels.

    Creates a point feature from each non nodata raster pixel, where
    the geometry for the point is the center of the pixel. A field
    'Value' is added to each point feature with the value from the pixel.
    The created point shapefile will use a spatial reference taking from
    the rasters projection.

    Parameters:
        raster_path (string): a filepath on disk of the raster to convert
            into a point shapefile.
        point_vector_path (string): a filepath on disk for where to save
            the shapefile. Must have a '.shp' extension.

    Returns:
        Nothing
    """
    # Shore raster to point shapefile
    pixel_size = natcap.invest.pygeoprocessing_0_3_3.geoprocessing.get_cell_size_from_uri(
        raster_path)
    nodata = natcap.invest.pygeoprocessing_0_3_3.geoprocessing.get_nodata_from_uri(
        raster_path)
    # [top_left_x, w-e pixel res, 0, top left y, 0, n-s pixel res negative]
    raster_gt = natcap.invest.pygeoprocessing_0_3_3.geoprocessing.get_geotransform_uri(
        raster_path)

    # Projection to use for new point shapefile
    proj = natcap.invest.pygeoprocessing_0_3_3.geoprocessing.get_dataset_projection_wkt_uri(
        raster_path)
    spat_ref = osr.SpatialReference()
    spat_ref.ImportFromWkt(proj)

    if os.path.isfile(point_vector_path):
        driver = ogr.GetDriverByName('ESRI Shapefile')
        driver.DeleteDataSource(point_vector_path)

    driver = ogr.GetDriverByName('ESRI Shapefile')
    ds = driver.CreateDataSource(point_vector_path)
    layer_name = str(os.path.basename(
                     os.path.splitext(point_vector_path)[0]))
    layer = ds.CreateLayer(layer_name, spat_ref, ogr.wkbPoint)
    field_defn = ogr.FieldDefn('Value', ogr.OFTReal)
    layer.CreateField(field_defn)

    for data, block in natcap.invest.pygeoprocessing_0_3_3.geoprocessing.iterblocks(
        raster_path, band_list=[1]):

        # The upper left column coordinate of the block
        x_start = raster_gt[0] + (data['xoff'] * pixel_size)
        # The upper left row coordinate of the block
        y_start = raster_gt[3] - (data['yoff'] * pixel_size)

        # We ultimately only want to act on shore pixels, which are the only
        # non-nodata pixels in the matrix
        shore_pixels_mask = (block != nodata)

        # If there are no shore pixels at all in this block, skip the block
        # entirely
        if np.all(~shore_pixels_mask):
            continue

        # Use the shore pixels mask to only iterate over those row, col
        # indices that represent shore pixels
        indices = np.indices(block.shape)
        for shore_row_idx, shore_col_idx in np.nditer(
            (indices[0][shore_pixels_mask], indices[1][shore_pixels_mask])):

            # Get the row, col coordinates of the middle of the cell
            # and add as the new Point Geometry
            geom = ogr.Geometry(ogr.wkbPoint)
            geom.AddPoint_2D(
                x_start + (shore_col_idx * pixel_size) + (pixel_size / 2.),
                (y_start - (shore_row_idx * pixel_size) - (pixel_size / 2.)))

            feature = ogr.Feature(layer.GetLayerDefn())
            layer.CreateFeature(feature)

            field_index = feature.GetFieldIndex('Value')
            # Set the new field to the value of the raster cell
            feature.SetField(
                field_index, float(block[shore_row_idx][shore_col_idx]))

            feature.SetGeometryDirectly(geom)
            layer.SetFeature(feature)

def nearest_vector_neighbor(neighbors_path, point_path, inherit_field):
    """Inherit a field value from the closest shapefile feature.

    Each point in 'point_path' will inherit field 'inherit_field' from the
    closest feature in 'neighbors_path'. Uses an rtree to build up a spatial
    index of 'neighbor_path' bounding boxes to find nearest points.

    Parameters:
        neighbors_path (string): a filepath on disk to a shapefile that has
            at least one field called 'inherit_field'
        point_path (string): a filepath on disk to a shapefile. A field
            'inherit_field' will be added to the point features. The value
            of that field will come from the closest feature's field in
            'neighbors_path'
        inherit_field (string): the name of the field in 'neighbors_path' to
            pass along to 'point_path'.

    Returns:
        Nothing
    """
    neighbor_ds = gdal.OpenEx(neighbors_path)
    point_ds = gdal.OpenEx(point_path, 1)

    neighbor_layer = neighbor_ds.GetLayer()
    point_layer = point_ds.GetLayer()
    # Create new field in the point shapefile, which should also be in the
    # neighbor shapefile field definitions
    field_defn = ogr.FieldDefn(inherit_field, ogr.OFTReal)
    point_layer.CreateField(field_defn)
    # Create a rtree spatial index for neighbor bounding boxes
    spatial_index = index.Index()

    for neighbor_feat in neighbor_layer:
        neighbor_FID = neighbor_feat.GetFID()
        neighbor_geom = neighbor_feat.GetGeometryRef()
        # Get the field value for the neighbor feature to possibly store in
        # the point shapefile
        neighbor_val = neighbor_feat.GetField(inherit_field)
        # Use Shapely to create nice geometry with proper bounding box format
        shapely_geom = shapely.wkb.loads(neighbor_geom.ExportToWkb())
        # Insert feature index, feature bounding box, and field value as object
        spatial_index.insert(
            neighbor_FID, shapely_geom.bounds, obj=neighbor_val)

    for point_feat in point_layer:
        point_geom = point_feat.GetGeometryRef()
        # Get the 6 nearest bounding boxes to the point geometry,
        # returning object param as well
        # NOTE: using a default of 6 to help ensure we get the actual closest
        hits = spatial_index.nearest(
            (point_geom.GetX(), point_geom.GetY(),
             point_geom.GetX(), point_geom.GetY()), 6, objects=True)

        shortest_dist = -1
        field_value = 0
        # Iterating over the 6 nearest bbox because it's possible the nearest
        # bbox is not necessarily the closest feature geometry to the point
        for data in hits:
            neighbor_feat = neighbor_layer.GetFeature(data.id)
            # Calculate actual distance from features to ensure we get the
            # closest pair
            cur_dist = point_geom.Distance(neighbor_feat.GetGeometryRef())
            if shortest_dist == -1:
                shortest_dist = cur_dist
                field_value = data.object
            if cur_dist < shortest_dist:
                shortest_dist = cur_dist
                field_value = data.object

        out_index = point_feat.GetFieldIndex(inherit_field)
        point_feat.SetField(out_index, field_value)

        point_layer.SetFeature(point_feat)

# TODO: update this docstring
def compute_relief_rank(args):
    """ Compute the relief index as is described in InVEST's user guide.

        Inputs:
            - args['relief_uri']: uri to an elevation dataset.
            - args['aoi_uri']: uri to the datasource of the region of interest.
            - args['landmass_uri']: uri to the landmass datasource where land is
              1 and sea is 0.
            - args['spread_radius']: if the coastline from the geomorphology i
                doesn't match the land polygon's shoreline, we can increase the
                overlap by 'spreading' the data from the geomorphology over a
                wider area. The wider the spread, the more ranking data overlaps
                with the coast. The spread is a convolution between the
                geomorphology ranking data and a 2D gaussian kernel of area
                (2*spread_radius+1)^2. A radius of zero reduces the kernel to
                the scalar 1, which means no spread at all.
            - args['spread_radius']: how much the shore coast is spread to match
                the relief's coast.
            - args['shore_raster_uri']: URI to the shore tiff dataset.
            - args['cell_size']: granularity of the rasterization.
            - args['intermediate_directory']: where intermediate files are
                stored

        Output:
            - Return R_relief as described in the user manual.
            - A rastrer file called relief.tif"""
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    prefix = args['prefix'] + 'RELIEF'
    # Dictionary that contains the data's uri indexed by name
    data_uri = {}
    # Get the land elevation and land coverage from relief. Sea elevation is 0
    def adjust_land_elevation(nodata):
        def compute(val):
            result = val if val is not nodata else 0.
            return float(result) if result > 0 else 0.
        return compute
    bathymetry_raster = gdal.OpenEx(args['relief_raster_uri'])
    bathymetry_band = bathymetry_raster.GetRasterBand(1)
    bathymetry_nodata = bathymetry_band.GetNoDataValue()
    bathymetry_band = None
    bathymetry_raster = None
    adjust_bathymetry = adjust_land_elevation(bathymetry_nodata)
    data_uri['land_elevation'] = os.path.join(intermediate_directory, prefix + \
        '_01_land_elevation.tif')
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets([args['relief_raster_uri']], \
    adjust_bathymetry, data_uri['land_elevation'], gdal.GDT_Float32, 0., \
    args['cell_size'], "intersection", aoi_uri = args['aoi_uri'])
    # Load land elevation
    land_elevation_raster = None
    land_elevation_raster = gdal.OpenEx(data_uri['land_elevation'])
    land_elevation = land_elevation_raster.GetRasterBand(1).ReadAsArray()
    land_elevation_raster = None
    # Create convolution kernel
    kernel = disc_kernel(args['elevation_averaging_radius'] /args['cell_size'])
    kernel = kernel.astype(float) / float(np.sum(kernel))
    # Convolve the land elevation with the kernel
    elevation_average = \
        sp.signal.convolve2d(land_elevation, kernel, mode = 'same')
    data_uri['elevation_average'] = os.path.join(intermediate_directory, \
    prefix + '_02_elevation_average.tif')
    # Save the convolved elevation
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.new_raster_from_base_uri( \
        args['aoi_raster_uri'], data_uri['elevation_average'], \
        'GTiff', 0., gdal.GDT_Float32)
    elevation_average_raster = \
        gdal.OpenEx(data_uri['elevation_average'], gdal.GA_Update)
    elevation_average_raster.GetRasterBand(1).WriteArray(elevation_average)
    elevation_average_raster = None
    # Convolve the land coverage with the kernel
    land_raster = gdal.OpenEx(args['landmass_raster_uri'])
    land_coverage = land_raster.GetRasterBand(1).ReadAsArray()
    land_raster = None
    land_proportion = sp.signal.convolve2d(land_coverage, kernel, mode='same')
    # Save the land proportion in the raster land_proportion
    data_uri['land_proportion'] = os.path.join(intermediate_directory, \
    prefix + '_03_land_proportion.tif')
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.new_raster_from_base_uri( \
        args['aoi_raster_uri'], data_uri['land_proportion'], \
        'GTiff', 0., gdal.GDT_Float32)
    land_proportion_raster = \
        gdal.OpenEx(data_uri['land_proportion'], gdal.GA_Update)
    land_proportion_band = land_proportion_raster.GetRasterBand(1)
    land_proportion_band.Fill(1.)
    land_proportion_band.FlushCache()
    land_proportion_band.WriteArray(land_proportion)
    land_proportion_band = None
    land_proportion_raster = None
    # Compute average land height along the coast
    min_proportion = np.unique(kernel)[0]
    def compute_land_proportion(elevation, proportion, shore):
        if shore:
            if proportion:
                return elevation/proportion
            else:
                return elevation/min_proportion
        else:
            return -1.0
    data_uri['average_land_height'] = \
    os.path.join(intermediate_directory, prefix + '_04_average_land_height.tif')
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets(
        [data_uri['elevation_average'], \
        data_uri['land_proportion'], args['shore_raster_uri']], \
        compute_land_proportion, \
        data_uri['average_land_height'], gdal.GDT_Float32, -1., args['cell_size'], \
        "intersection", aoi_uri = args['aoi_uri'])
    # Rank the shore segments
    land_height_raster = gdal.OpenEx(data_uri['average_land_height'])
    land_height = land_height_raster.GetRasterBand(1).ReadAsArray()
    land_height_raster = None
    shore_points = np.where(land_height > -1.)
    data_uri['relief'] = \
        os.path.join(intermediate_directory, args['prefix'] + 'relief.tif')
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.new_raster_from_base_uri( \
        args['aoi_raster_uri'], data_uri['relief'], \
        'GTiff', 0., gdal.GDT_Float32)
    relief_raster = gdal.OpenEx(data_uri['relief'], gdal.GA_Update)
    relief = relief_raster.GetRasterBand(1).ReadAsArray()
    relief[shore_points] = rank_shore(-land_height[shore_points], 5) + 1
    relief_band = relief_raster.GetRasterBand(1)
    relief_band.FlushCache()
    relief_band.WriteArray(relief)
    relief_band = None
    relief_raster = None
    return data_uri

# TODO: update doctring!
def compute_wind_exposure(args):
    """ Compute the wind exposure for every shore segment as in equation 4.5

        Inputs:
            - args['climatic_forcing_uri']: uri to the wind information
              datasource
            - args['aoi_uri']: uri to the area of interest datasource
            - args['fetch_distances']: a dictionary of (point, list) pairs
              where point is a tuple of integer (row, col) coordinates and
              list is a maximal fetch distance in meters for each fetch sector.
            - args['fetch_depths']: same dictionary as fetch_distances, but
              list is a maximal fetch depth in meters for each fetch sector.
            - args['cell_size']: granularity of the rasterization.
            - args['intermediate_directory']:where intermediate files are stored
            - args['prefix']: string

        Outputs:
            - data_uri: dictionary of the uri of all the files created in the
              function execution
        File description:
            - REI.tif: combined REI value of the wind exposure index for all
              sectors along the shore.

            - For each equiangular fetch sector n:
                - REI_n.tif: per-sector REI value (U_n * P_n * F_n)."""
    nodata = 0
    sector_count = SECTOR_COUNT
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    prefix = args['prefix']
    # Dictionary that contains the data's uri indexed by name
    data_uri = {}
    # Variable initializations
    sector_range = range(sector_count)
    sectors_deg = np.array(sector_range, dtype=int) *360 / sector_count
    basename = prefix + 'WIND'
    extension = '.tif'  # Filename extension
    # Compute the REI for each sector
    REI_uri_list = []
    for sector in range(sector_count):
        REI_uri = os.path.join(intermediate_directory, basename + '_REI_' + \
        str(sectors_deg[sector]) + extension)
        # Shore_uri is used as a mask to leave 0s in the background
        input_uri_list = [args['REI_V'][sector], args['REI_PCT'][sector], \
        args['fetch_distance_uris'][sector], args['shore_raster_uri']]
        natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets(input_uri_list, \
        lambda x, y, z, mask: x*y*z if mask > 0 else nodata,\
        REI_uri, gdal.GDT_Float32, nodata, args['cell_size'], 'intersection')
        REI_uri_list.append(REI_uri)
        # Copy REI_V and REI_PCT to the wind subdirectory:
        shutil.copy(args['REI_V'][sector], intermediate_directory)
        shutil.copy(args['REI_PCT'][sector], intermediate_directory)
    # Sum up the result
    REI_uri = os.path.join(intermediate_directory, basename + extension)
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets(REI_uri_list, lambda *x: sum(x), \
    REI_uri, gdal.GDT_Float32, nodata, args['cell_size'], 'intersection', \
    aoi_uri = args['aoi_uri'])
    # Use percentiles to assign vulnerability index
    wind_exposure_uri = \
    os.path.join(intermediate_directory, prefix + 'wind_exposure.tif')
    # Non-shore is zero, the rest is positive
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets([args['shore_raster_uri'], REI_uri], \
    lambda x, y: x*y, wind_exposure_uri, gdal.GDT_Float32, nodata, \
    args['cell_size'], 'intersection')
    shore_raster = gdal.OpenEx(wind_exposure_uri, gdal.GA_Update)
    shore_array = shore_raster.GetRasterBand(1).ReadAsArray()
    shore_points = np.where(shore_array != nodata)
    shore_array[shore_points] = rank_shore(shore_array[shore_points], 5) + 1
    shore_raster.GetRasterBand(1).WriteArray(shore_array)
    data_uri['wind_exposure'] = wind_exposure_uri
    # done
    return data_uri

def assign_sheltered_segments(exposure_raster_uri, raster_uri,
    output_raster_uri):
    """ Propagate values from 'sources' across a surface defined by 'mask' in a
        breadth-first-search manner.

        Inputs:
            -exposure_raster_uri: URI to the GDAL dataset that we want to process
            -mask: a numpy array where 1s define the area across which we want
                to propagate the values defined in 'sources'.
            -sources: a tuple as is returned by numpy.where(...) of coordinates
                of where to pick values in 'raster_uri' (a source). They are the
                values we want to propagate across the area defined by 'mask'.
            -output_raster_uri: URI to the GDAL dataset where we want to save
            the array once the values from source are propagated.

        Returns: nothing.

        The algorithm tries to spread the values pointed by 'sources' to every
        of the 8 immediately adjascent pixels where mask==1. Each source point
        is processed in sequence to ensure that values are propagated from the
        closest source point.
        If a connected component of 1s in 'mask' does not contain any source,
        its value remains unchanged in the output raster."""
    def find_exposure_boundaries(exposure_array, no_data):
        """Finds sheltered shore segments bordering exposed ones."""
        mask = np.where(exposure_array != no_data)
        exposure_array[exposure_array == no_data] = 0
        kernel = np.array([[1, 1, 1], [1, -9, 1], [1, 1, 1]])

        shore = sp.signal.convolve2d(exposure_array, kernel, mode='same')
        boundary_indices = np.where(shore[mask] > 0)[0]

        return (mask[0][boundary_indices], mask[1][boundary_indices])
    #LOGGER.info('Assigning sheltered segment ranks using exposed segments.')
    # Extract the mask, i.e. the shorteline over which to propagate the values
    exposure_raster = gdal.OpenEx(exposure_raster_uri)
    band = exposure_raster.GetRasterBand(1)
    no_data = band.GetNoDataValue()
    exposure_array = band.ReadAsArray()

    mask = np.where(exposure_array == 0)

    sources = find_exposure_boundaries(exposure_array, no_data)
    if not sources:
        return
    raster = gdal.OpenEx(raster_uri)
    nodata = raster.GetRasterBand(1).GetNoDataValue()
    array = np.copy(raster.GetRasterBand(1).ReadAsArray())

    # Compute the values that should be propagated
    for n in range(sources[0].size):
        i = sources[0][n]
        j = sources[1][n]
        # Slice a 3x3 array around the point
        x = np.array([i-1, i, i+1])
        y = np.array([j-1, j, j+1])
        neighborhood = np.vstack(np.meshgrid(x, y)).reshape(2, -1)
        neighborhood = (neighborhood[0], neighborhood[1])
        # Extract points where values should be propagated
        # If the cell is bordered by exposed segments with different ranks,
        # take the highest vulnerability
        try:
            array[i, j] = np.amax(array[neighborhood])
        except IndexError:
            # If we're indexing off the edge of the array, skip this point
            # I'm sure there's a more intelligent way to handle this edge case,
            # but avoiding it altogether is easier.
            continue

    mask_array = np.zeros_like(array)
    mask_array[mask] = 1
    mask_array[sources] = 0 # Don't need to propagate here
    max_array_row = array.shape[0] - 1
    max_array_col = array.shape[1] - 1

    # Stop when there are no more pixels sources to propagate across the mask
    # However, there could be remaining 1s in mask_array if some cells are
    # connected to source-free components (components with no source on them).
    while sources[0].size > 0:
        new_sources = () # sources to propagate on the next round
        for n in range(sources[0].size):
            i = sources[0][n]
            j = sources[1][n]
            value = array[i, j] # This is the value to propagate
            # Slice a 3x3 array around the point and if the point is at the
            # raster edge, min and max ensures the array fits within the raster
            x = np.array([max(0, i-1), i, min(max_array_row, i+1)])
            y = np.array([max(0, j-1), j, min(max_array_col, j+1)])
            neighborhood = np.vstack(np.meshgrid(x, y)).reshape(2, -1)
            neighborhood = (neighborhood[0], neighborhood[1])
            # Extract points where values should be propagated
            #print('mask_array', mask_array.shape)
            #print('neighborhood', neighborhood[0], neighborhood[1])
            point_indices = np.where(mask_array[neighborhood] == 1)[0]
            points = \
            (neighborhood[0][point_indices], neighborhood[1][point_indices])
            # Assign the values to the array
            array[points] = value
            # We're done with these points, remove them from mask
            mask_array[points] = 0
            # These are the new values to propagate next round
            if not new_sources:
                new_sources = points
            else:
                first = np.concatenate([new_sources[0], points[0]])
                second = np.concatenate([new_sources[1], points[1]])
                new_sources = (first, second)
        # Update sources with the new ones we just found
        sources = new_sources

    # If there are remaining 1s in mask_array, assign the value of the closest
    # exposed shore segment
    exposed_segments = np.where(exposure_array == 1)
    unassigned_shore = np.where(mask_array == 1)
    if unassigned_shore[0].size > 0:
        # Initialize the KD tree with exposed shore segments
        tree = spatial.KDTree(zip(exposed_segments[0], exposed_segments[1]))
        # Find the exposed segments closest to the unassigned shore
        _, exposed_indices = \
            tree.query(zip(unassigned_shore[0], unassigned_shore[1]))
        # Fill the unassigned segments with the values from the closest exposed
        # shore segment:
        # tree.data[exposed_indices]: coordinates of the closest exposed segments
        # zip(*tree.data[...]): coordinates formatted as in np.where(...)
        # array[zip(...)]: values of the closest exposed shore segments, ordered as
        # in unassigned_shore
        array[unassigned_shore] = array[zip(*tree.data[exposed_indices])]

    # Save values to the output raster
    new_raster = \
        natcap.invest.pygeoprocessing_0_3_3.geoprocessing.new_raster_from_base(raster, output_raster_uri, \
    	'GTIFF', nodata, gdal.GDT_Float32)
    new_raster.GetRasterBand(1).WriteArray(array)
    return

# TODO: Update docstring!
def compute_wave_exposure(args):
    """ Compute the wind exposure for every shore segment

        Inputs:
            - args['climatic_forcing_uri']: uri to wave datasource
            - args['aoi_uri']: uri to area of interest datasource
            - args['fetch_distances']: a dictionary of (point, list) pairs
              where point is a tuple of integer (row, col) coordinates and
              list is a maximal fetch distance in meters for each fetch sector.
            - args['fetch_depths']: same dictionary as fetch_distances, but
              list is a maximal fetch depth in meters for each fetch sector.
            - args['cell_size']: cell size in meters (integer)
            - args['H_threshold']: threshold (double) for the H function (eq. 7)
            - args['intermediate_directory']: uri to the directory that
              contains the intermediate files

        Outputs:
            - data_uri: dictionary of the uri of all the files created in the
              function execution
        Detail of files:
            - A file called wave.tif that contains the wind exposure index along
              the shore.
            - For each equiangular fetch sector k:
                - F_k.tif: per-sector fetch value (see eq. 6).
                - H_k.tif: per-sector H value (see eq. 7)
                - E_o_k.tif: per-sector average oceanic wave power (eq. 6)
                - E_l_k.tif: per-sector average wind-generated wave power (eq.9)
                - E_w_k.tif: per-sector wave power (eq.5)
                - E_w.tif: combined wave power."""
    shore = gdal.OpenEx(args['shore_raster_uri'])
    shore_array = shore.GetRasterBand(1).ReadAsArray()
    shore_points = np.where(shore_array > 0.)
    shore_array = None

    sector_count = SECTOR_COUNT
    H_threshold = args['H_threshold']
    intermediate_directory = \
        os.path.join(args['intermediate_directory'], args['subdirectory'])
    # Dictionary that contains the data's uri indexed by name
    data_uri = {}
    # Variable initializations
    sector_range = range(sector_count)
    sectors_deg = np.array(sector_range, dtype=int) *360 / sector_count
    prefix = args['prefix']
    extension = '.tif'  # Filename extension

    # E_o needs the step function H. This function initializes E_o with the
    # function H
    def set_H_for_E_o(H):
        def E_o(fetch, WavP, WavPPCT, mask):
            if mask == 0.:
                return -1.
            return max(H(fetch) * WavP * WavPPCT, 0)
        return E_o

    # E_o needs the function H. Set H first, and use it to initialize E_o
    # Set H threshold, in km
    H = set_H_threshold(H_threshold)
    E_o = set_H_for_E_o(H)

    # Compute the oceanic wave energy for each sector
    Eo_uri_list = [args['shore_raster_uri'], args['shore_exposure']]
    for sector in range(sector_count):
        Eo_uri = os.path.join(intermediate_directory, prefix + 'E_o_' + \
        str(sectors_deg[sector]) + extension)
        # Shore_uri is used as a mask to leave 0s in the background
        input_uri_list = [args['fetch_distance_uris'][sector], \
        args['WavP'][sector], args['WavPPCT'][sector], args['shore_raster_uri']]
        natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets(input_uri_list, E_o, \
        Eo_uri, gdal.GDT_Float32, -1., args['cell_size'], 'intersection')
        Eo_uri_list.append(Eo_uri)
    # Sum the result
    Eo_uri = os.path.join(intermediate_directory, \
    prefix +'oceanic_wave_power' +extension)
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets( \
        Eo_uri_list, lambda *x: x[1] * sum(x[2:]) if x[0] else -1., \
        Eo_uri, gdal.GDT_Float32, -1., args['cell_size'], \
        'intersection', aoi_uri = args['aoi_uri'])
    data_uri['E_o'] = Eo_uri

    # TODO: docstring!
    bad_wind_value = [False]
    def compute_wave_height(Un, Fn, dn, mask):
        if mask == 0.:
            return -1.
        if Un < 1.:
            bad_wind_value[0] = True
            Un = 1.
        dn = -dn
        ds = 9.81*dn/Un**2
        Fs = 9.81*Fn/Un**2
        A = np.tanh(0.343*ds**1.14)
        if Fs < 0:
            print('Fs=', Fs, 'Fn=', Fn)
        B = np.tanh(4.41e-4*Fs**0.79/A)
        H_n = 0.24*Un**2/9.81*(A*B)**0.572
        return H_n

    # Compute wave height for each sector
    wave_height_list = []
    for sector in range(sector_count):
        #print("sector", sector)
        #print("fetch_distance_uri", args['fetch_distance_uris'][sector])
        bad_wind_value[0] = False
        uri = os.path.join(intermediate_directory, \
        prefix + 'wave_height_' + str(sectors_deg[sector]) + extension)
        # Shore_uri is used as a mask
        input_uri_list = [args['REI_V'][sector], \
        args['fetch_distance_uris'][sector],
        args['fetch_depth_uris'][sector], args['shore_raster_uri']]
        natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets(input_uri_list, compute_wave_height, \
        uri, gdal.GDT_Float32, -1., args['cell_size'], 'intersection')
        wave_height_list.append(uri)
        if bad_wind_value[0]:
            LOGGER.warning('One or more wind speeds <= 0 and were set to 1.')
    args['wave_heights'] = wave_height_list

    # TODO: docstring!
    def compute_wave_period(Un, Fn, dn, mask):
        if mask == 0.:
            return -1.
        if Un < 1.:
            Un = 1.
        dn = -dn
        ds = 9.81*dn/Un**2
        Fs = 9.81*Fn/Un**2
        A = np.tanh(0.1*ds**2.01)
        B = np.tanh(2.77e-7*Fs**1.45/A)
        return 7.69*Un/9.81*(A*B)**0.187

    # Compute wave period for each sector
    wave_period_list = []
    for sector in range(sector_count):
        uri = os.path.join(intermediate_directory, prefix + \
        'wave_period_' + str(sectors_deg[sector]) + extension)
        # Shore_uri is used as a mask
        input_uri_list = [args['REI_V'][sector], \
        args['fetch_distance_uris'][sector],
        args['fetch_depth_uris'][sector], args['shore_raster_uri']]
        natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets(input_uri_list, compute_wave_period, \
        uri, gdal.GDT_Float32, -1., args['cell_size'], 'intersection')
        wave_period_list.append(uri)
    args['wave_periods'] = wave_period_list

    def E_l(wave_height, wave_period, REI_PCT, mask):
        if mask == 0:
            return -1.
        return 0.5 * wave_height**2 * wave_period * REI_PCT

    # Compute local wave power for each sector
    # Adding shore exposure to the uri list to set sheltered segments to 0
    El_uri_list = [args['shore_raster_uri']]
    for sector in range(sector_count):
        uri = os.path.join(intermediate_directory, prefix + \
        'E_l_' + str(sectors_deg[sector]) + extension)
        # Shore_uri is used as a mask
        input_uri_list = [wave_height_list[sector], wave_period_list[sector], \
        args['REI_PCT'][sector], args['shore_raster_uri']]
        natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets(input_uri_list, E_l, \
        uri, gdal.GDT_Float32, -1., args['cell_size'], 'intersection')
        El_uri_list.append(uri)

    # Combine the result: sum the values for all sectors
    El_uri = os.path.join(
        intermediate_directory, prefix +'local_wave_power' + extension)
    # Sum for all the sectors
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets( \
        El_uri_list, \
        lambda *x: sum(x[1:]) if x[0] else -1., \
        El_uri, gdal.GDT_Float32, -1., args['cell_size'], \
    'intersection', aoi_uri = args['aoi_uri'])
    data_uri['E_l'] = El_uri
    # Take the maximum of the two
    Ew_uri = os.path.join(intermediate_directory, prefix +'wave_power.tif')
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_datasets([Eo_uri, El_uri, args['shore_raster_uri']], \
    lambda x, y, z: max(x, y) if z else -1., Ew_uri, gdal.GDT_Float32, -1., \
    args['cell_size'], 'intersection', aoi_uri = args['aoi_uri'])
    data_uri['E_w'] = Ew_uri
    # Use percentiles to assign vulnerability index
    ranked_Ew_uri = \
    os.path.join(intermediate_directory, args['prefix'] +'wave_exposure.tif')
    shutil.copy(Ew_uri, ranked_Ew_uri)
    shore_raster = gdal.OpenEx(ranked_Ew_uri, gdal.GA_Update)
    shore_array = shore_raster.GetRasterBand(1).ReadAsArray()
    shore_points = np.where(shore_array != -1.)
    shore_array[shore_points] = rank_shore(shore_array[shore_points], 5) + 1
    shore_raster.GetRasterBand(1).WriteArray(shore_array)
    data_uri['wave_exposure'] = ranked_Ew_uri
    # done
    return data_uri

def rank_by_quantiles(X, bin_count):
    """ Tries to evenly distribute elements in X among 'bin_count' bins.
        If the boundary of a bin falls within a group of elements with the same
        value, all these elements will be included in that bin.
        Inputs:
            -X: a 1D numpy array of the elements to bin
            -bin_count: the number of bins

        Returns the bin boundaries ready to be used by numpy.digitize"""
    I = np.argsort(X)
    ns = [X.size*n/bin_count for n in range(bin_count)]
    Is = I[ns]
    bin_boundaries = X[Is]
    #bin_boundaries = [X[I[X.size/bin_count*n]] for n in range(bin_count)]
    #bin_boundaries.append(X[I[-1]] + 1)
    bin_boundaries = np.append(bin_boundaries, X[I[-1]] + 1)
    # Detect if values span more than one rank, and warn the user.
    #difference = np.setdiff1d(bin_boundaries, np.unique(bin_boundaries))
    #if difference.size:
    #    bin_count = np.unique(bin_boundaries) - 1
    #    LOGGER.warning('The following values span more than one rank: ' + \
    #    str(difference) + '. There will be ' + str(bin_count) + ' bins.')
    #    bin_boundaries = np.unique(bin_boundaries)
    return bin_boundaries

def rank_shore(X, bin_count):
    """ Assign a rank based on natural breaks (Jenks natural breaks for now).

        Inputs:
            - X: a numpy array with the lements to be ranked
            - bins: the number of ranks (integer)

        Outputs:
            - output: a numpy array with rankings in the interval
                [0, bin_count-1] that correspond to the elements of X
                (rank of X[i] == outputs[i])."""
    try:
        Y = np.copy(X) - np.amin(X)
    except ValueError:
        # occurs if len(X) == 0.
        # This happens when shoreline exposure does not detect any exposed
        # shore segments.
        raise ValueError('No exposed shoreline segments found. This could be '
                         'from a too-high depth threshold or a too-low '
                         'exposure proportion')

    unique_values = np.unique(Y)
    unique_count = len(unique_values)
    # Adjust the bins according to the number of different values
    if unique_count > bin_count:
        # More unique values than the number of categories? rank them
        #bins = np.array(get_jenks_breaks_fast(np.copy(Y), 5))
        bins = np.array(rank_by_quantiles(Y, 5))
    else:
        # Otherwise use unique_values for the bins
        # Add a large constant at the end so the bins are compatible with
        # numpy.digitize
        bins = np.append(unique_values, unique_values[-1] + 1)
    output = np.digitize(Y, bins)
    output -= 1

    return output


def enumerate_shapefile_fields(shapefile_uri):
    """Enumerate all the fielfd in a shapefile.

        Inputs:
            -shapefile_uri: uri to the shapefile which fields have to be
            enumerated

        Returns a nested list of the field names in the order they are stored
        in the layer, and groupped per layer in the order the layers appear.
    """
    message = shapefile_uri + "' doesn't point to a file."
    assert os.path.isfile(shapefile_uri), message
    shapefile = gdal.OpenEx(shapefile_uri)
    message = "OGR can't open " + shapefile_uri
    assert shapefile is not None, message
    layer_count = shapefile.GetLayerCount()
    names = [] # names are organized by layer
    for l in range(layer_count):
        names.append([])
        layer = shapefile.GetLayer(l)
        feature = layer.GetFeature(0)
        field_count = feature.GetFieldCount()
        for f in range(field_count):
            field_defn = feature.GetFieldDefnRef(f)
            names[l].append(field_defn.GetNameRef())
    return names

# TODO: write a unit test for this function
def get_field(field_name, shapefile, case_sensitive = True):
    """Return the field in shapefile that corresponds to field_name,
        None otherwise.

        Inputs:
            - field_name: string to look for.
            - shapefile: where to look for the field.
            - case_sensitive: indicates whether the case is relevant when
            comparing field names

        Output:
            - the field name in the shapefile that corresponds to field_name,
            None otherwise."""
    layer_count = shapefile.GetLayerCount()
    for l in range(layer_count):
        layer = shapefile.GetLayer(l)
        feature_count = layer.GetFeatureCount()
        assert feature_count > 0
        feature = layer.GetFeature(0)
        field_count = feature.GetFieldCount()
        for f in range(field_count):
            field_defn = feature.GetFieldDefnRef(f)
            if case_sensitive:
                if field_defn.GetNameRef() == field_name:
                    return field_defn.GetNameRef()
            # if not case sensitive, convert everything to lower case
            else:
                if field_defn.GetNameRef().lower() == field_name.lower():
                    return field_defn.GetNameRef()
    return None

def has_field(field_name, shapefile, case_sensitive = True):
    return get_field(field_name, shapefile, case_sensitive) is not None

# TODO: write a unit test for this function
def get_layer_and_index_from_field_name(field_name, shapefile):
    """Given a field name, return its layer and field index.
        Inputs:
            - field_name: string to look for.
            - shapefile: where to look for the field.

        Output:
            - A tuple (layer, field_index) if the field exist in 'shapefile'.
            - (None, None) otherwise."""
    # Look into every layer
    layer_count = shapefile.GetLayerCount()
    for l in range(layer_count):
        layer = shapefile.GetLayer(l)
        # Make sure the layer is not empty
        feature_count = layer.GetFeatureCount()
        if feature_count > 0:
            feature = layer.GetFeature(0)
            # Enumerate every field
            field_count = feature.GetFieldCount()
            for f in range(field_count):
                field_defn = feature.GetFieldDefnRef(f)
                if field_defn.GetNameRef() == field_name:
                    return (l, f)
    # Didn't find fields, enumerate their names and the feature counts
    layer_count = shapefile.GetLayerCount()
    for l in range(layer_count):
        layer = shapefile.GetLayer(l)
        feature_count = layer.GetFeatureCount()
        print("Layer " + str(l) + " has " + str(feature_count) + " features.")
        if feature_count > 0:
            feature = layer.GetFeature(0)
            # Enumerate every field
            field_count = feature.GetFieldCount()
            print('fields:')
            for f in range(field_count):
                field_defn = feature.GetFieldDefnRef(f)
                print(field_defn.GetNameRef())
    # Nothing found
    return (None, None)

def combined_rank(R_k):
    """Compute the combined habitats ranks as described in equation (3)

        Inputs:
            - R_k: the list of ranks

        Output:
            - R_hab as decribed in the user guide's equation 3."""
    R = 4.8 -0.5 *math.sqrt( (1.5 *max(5-R_k))**2 + \
                    sum((5-R_k)**2) -(max(5-R_k))**2)
    if R <=0:
        print('rank', R, 'R_k', R_k)
    return R

def disc_kernel(r):
    """Create a (r+1)^2 disc-shaped array filled with 1s where d(i-r,j-r) <= r

        Input: r, the kernel radius. r=0 is a single scalar of value 1.

        Output: a (r+1)x(r+1) array with:
                - 1 if cell is closer than r units to the kernel center (r,r),
                - 0 otherwise.

            Distances are Euclidean."""
    # Create a grid of an evenly-spaced datapoints of diameter r * 2 + 1
    [X, Y] = np.mgrid[0:(r*2+1), 0:(r*2+1)]
    # Compute an array of 1s where the distance to the center (r,r) < r
    kernel = (r**2 >= np.square(X-r)+np.square(Y-r)).astype(int)
    return kernel

def set_H_threshold(threshold):
    """ Return 0 if fetch is strictly below a threshold in km, 1 otherwise.

        Inputs:
            fetch: fetch distance in meters.

        Return:
            1 if fetch >= threshold (in km)
            0 if fetch  < threshold

        Note: conforms to equation 4.8 in the invest documentation."""
    def H(fetch):
        return np.array(fetch >= threshold).astype(int)

    return H

def raster_wkt(raster):
    """ Return the projection of a raster in the OpenGIS WKT format.

        Input:
            - raster: raster file

        Output:
            - a projection encoded as a WKT-compliant string."""
    return raster.GetProjection()

def shapefile_wkt(shapefile):
    """ Return the projection of a shapefile in the OpenGIS WKT format.

        Input:
            - raster: raster file

        Output:
            - a projection encoded as a WKT-compliant string."""
    layer = shapefile.GetLayer()
    sr = layer.GetSpatialRef()
    return sr.ExportToWkt()

# TODO: write a unit test for this function
def projections_match(projection_list, silent_mode = True):
    """Check that two gdal datasets are projected identically.
       Functionality adapted from Doug's
       biodiversity_biophysical.check_projections

        Inputs:
            - projection_list: list of wkt projections to compare
            - silent_mode: id True (default), don't output anything, otherwise
              output if and why some projections are not the same.

        Output:
            - False the datasets are not projected identically.
    """
    assert len(projection_list) > 1

    srs_1 = osr.SpatialReference()
    srs_2 = osr.SpatialReference()

    srs_1.ImportFromWkt(projection_list[0])

    for projection in projection_list:
        srs_2.ImportFromWkt(projection)

        if srs_1.IsProjected() != srs_2.IsProjected():
            if not silent_mode:
                message = \
                'Different projection: One of the Rasters is Not Projected'
                LOGGER.debug(message)
            return False
        if srs_1.GetLinearUnits() != srs_2.GetLinearUnits():
            #LOGGER.debug('Different proj.: Proj units do not match %s:%s', \
            #         srs_1.GetLinearUnits(), srs_2.GetLinearUnits())
            return False

        if srs_1.GetAttrValue("PROJECTION") != srs_2.GetAttrValue("PROJECTION"):
            #LOGGER.debug('Projections are not the same')
            return False
        # At this point, everything looks identical. look further into the
        # projection parameters:
        for parameter in ['false_easting', 'false_northing', \
            'standard_parallel_1', 'standard_parallel_2', \
            'latitude_of_center', 'longitude_of_center']:
            if srs_1.GetProjParm(parameter) != srs_2.GetProjParm(parameter):
                return False
    return True

# TODO: write a unit test for this function
def adjust_shapefile_to_aoi(data_uri, aoi_uri, output_uri,
    empty_raster_allowed=False):
    """Adjust the shapefile's data to the aoi, i.e.reproject & clip data points.

        Inputs:
            - data_uri: uri to the shapefile to adjust
            - aoi_uri: uir to a single polygon shapefile
            - base_path: directory where the intermediate files will be saved
            - output_uri: dataset that is clipped and/or reprojected to the
            aoi if necessary.
            - empty_raster_allowed: boolean flag that, if False (default),
              causes the function to break if output_uri is empty, or return
              an empty raster otherwise.

        Returns: output_uri
    """
    # Data and aoi are the same URIs, data is good as it is, return it.
    if data_uri == aoi_uri:
        return data_uri
    # Split the path apart from the filename
    head, tail = os.path.split(output_uri)
    # Split the file basename from the file extension
    base, _ = os.path.splitext(tail)
    # Open URIs and get the projections
    data = gdal.OpenEx(data_uri)
    message = "OGR Can't open " + data_uri
    assert data is not None, message
    aoi = gdal.OpenEx(aoi_uri)
    data_wkt = shapefile_wkt(data)
    aoi_wkt = shapefile_wkt(aoi)

    if projections_match([data_wkt, aoi_wkt]):
        # Same projections, just clip
        clip_datasource(aoi, data, output_uri)
    else:
        # Reproject the aoi to be in data's projection
        projected_aoi_uri = os.path.join(head, base + '_projected_aoi')
        # TODO: include this in raster utils
        # Removing output_uri if it already exists
        if os.path.isdir(projected_aoi_uri):
            shutil.rmtree(projected_aoi_uri)

        # Add extra code to catch a strange error in GDAL due to a discrepancy
        # between GDAL and ESRI wkt formats that cause the model to crash:
        # http://gis.stackexchange.com/questions/61823/ogr-projection-transformation-error

        for layer_index in xrange(aoi.GetLayerCount()):
            layer = aoi.GetLayerByIndex(layer_index)
            output_sr = osr.SpatialReference()
            output_sr.ImportFromWkt(data_wkt)
            original_sr = aoi.GetLayer(0).GetSpatialRef()
            coord_trans = osr.CoordinateTransformation(original_sr, output_sr)
            try:
                coord_trans.TransformPoint(0, 0)
            except (TypeError, NotImplementedError) as e:
                message = 'The model cannot create an appropriate ' + \
                    "transformation between the current shapefile's " + \
                    "and the AOI's spatial references. This could " + \
                    "occur if your layer has a spatial reference " + \
                    "definition that is not strictly EPSG compliant, " + \
                    "which could be the case with some versions of ArcGIS."
                LOGGER.warning(message)
                raise

        natcap.invest.pygeoprocessing_0_3_3.geoprocessing.reproject_datasource(aoi, data_wkt, projected_aoi_uri)
        # Clip all the shapes outside the aoi
        out_uri = os.path.join(head, base + '_clipped')
        clip_datasource(gdal.OpenEx(projected_aoi_uri), data, out_uri)
        # Convert the datasource back to the original projection (aoi's)
        # TODO: include this in raster utils
        # Removing output_uri if it already exists
        if os.path.isdir(output_uri):
            shutil.rmtree(output_uri)
        natcap.invest.pygeoprocessing_0_3_3.geoprocessing.reproject_datasource(gdal.OpenEx(out_uri), aoi_wkt, \
        output_uri)
    # Ensure the resulting file's 1st layer is not empty
    out_shapefile = gdal.OpenEx(output_uri)
    out_layer = out_shapefile.GetLayer(0)
    out_feature_count = out_layer.GetFeatureCount()
    out_layer = None
    out_shapefile = None
    # Break if returning an empty raster is not allowed
    if not empty_raster_allowed:
        message = 'Error: first layer of ' + output_uri + ' is empty. Are ' + \
        data_uri + ' and ' + aoi_uri + ' mis-aligned?'
        assert out_feature_count > 0, message
    return output_uri

def clip_datasource(aoi_ds, orig_ds, output_uri):
    """Clip an OGR Datasource of geometry type polygon by another OGR Datasource
        geometry type polygon. The aoi_ds should be a shapefile with a layer
        that has only one polygon feature

        aoi_ds - an OGR Datasource that is the clipping bounding box
        orig_ds - an OGR Datasource to clip
        out_uri - output uri path for the clipped datasource

        returns - a clipped OGR Datasource """
    orig_layer = orig_ds.GetLayer()
    aoi_layer = aoi_ds.GetLayer()

    # If the file already exists remove it
    if os.path.isfile(output_uri):
        os.remove(output_uri)
    elif os.path.isdir(output_uri):
        shutil.rmtree(output_uri)

    # Create a new shapefile from the orginal_datasource
    output_driver = ogr.GetDriverByName('ESRI Shapefile')
    output_datasource = output_driver.CreateDataSource(output_uri)

    # Get the original_layer definition which holds needed attribute values
    original_layer_dfn = orig_layer.GetLayerDefn()

    # Create the new layer for output_datasource using same name and geometry
    # type from original_datasource as well as spatial reference
    output_layer = output_datasource.CreateLayer(
            original_layer_dfn.GetName(), orig_layer.GetSpatialRef(),
            original_layer_dfn.GetGeomType())

    # Get the number of fields in original_layer
    original_field_count = original_layer_dfn.GetFieldCount()

    # For every field, create a duplicate field and add it to the new
    # shapefiles layer
    for fld_index in range(original_field_count):
        original_field = original_layer_dfn.GetFieldDefn(fld_index)
        output_field = ogr.FieldDefn(
                original_field.GetName(), original_field.GetType())
        output_field.SetWidth(original_field.GetWidth())
        output_field.SetPrecision(original_field.GetPrecision())
        output_layer.CreateField(output_field)

    # Get the feature and geometry of the aoi
    aoi_feat = aoi_layer.GetFeature(0)
    aoi_geom = aoi_feat.GetGeometryRef()

    # Iterate over each feature in original layer
    missing_geometries = 0 # Count the number of features without a geometry
    total_geometries = 0
    for orig_feat in orig_layer:
        total_geometries += 1
        # Get the geometry for the feature
        orig_geom = orig_feat.GetGeometryRef()
        if orig_geom is None:
            missing_geometries += 1 # Will report the missing geometries later
            continue
        # Check to see if the feature and the aoi intersect. This will return a
        # new geometry if there is an intersection. If there is not an
        # intersection it will return an empty geometry or it will return None
        # and print an error to standard out
        intersect_geom = aoi_geom.Intersection(orig_geom)

        if not intersect_geom == None and not intersect_geom.IsEmpty():
            # Copy original_datasource's feature and set as new shapes feature
            output_feature = ogr.Feature(
                    feature_def=output_layer.GetLayerDefn())
            output_feature.SetGeometry(intersect_geom)
            # Since the original feature is of interest add it's fields and
            # Values to the new feature from the intersecting geometries
            for fld_index2 in range(output_feature.GetFieldCount()):
                orig_field_value = orig_feat.GetField(fld_index2)
                output_feature.SetField(fld_index2, orig_field_value)

            output_layer.CreateFeature(output_feature)
            output_feature = None
    # If missing geometries, report it
    if missing_geometries:
        message = 'Missing ' + str(missing_geometries) + \
        ' features out of' + str(total_geometries) + ' geometries in' + \
        output_uri
        LOGGER.warning(message)

    return output_datasource

# TODO: write a unit test for this function
# TODO: Update docstrings!
def save_fetch_distances(fetch, aoi_uri, cell_size, base_path, prefix = ''):
    """ Create dictionary of raster filenames of fetch F(n) for each sector n.

        Inputs:
            - wind_data: wind data points adjusted to the aoi
            - aoi: used to create the rasters for each sector
            - cell_size: raster granularity in meters
            - base_path: base path where the generated raster will be saved

        Output:
        A list of raster URIs corresponding to sectors of increasing angles
        where data points encode the sector's fetch distance for that point"""
    no_data = -32768.
    field_prefix = os.path.join(base_path, prefix + 'fetch_distance')
    extension = '.tif'

    # Save computed raster
    raster_list = []
    # Extract the # of sectors, assumed to start at angle 0 with identical span
    sector_count = 0
    fetch_keys = fetch.keys()
    for point in fetch_keys:
        sector_count = len(fetch[point])
        break
    # Build F(n), raster filename dictionary and save data to disk
    for sector in range(sector_count):
        angle = str(sector *360 /sector_count)
        # Open raster file
        filename = field_prefix + '_' + angle + extension
        raster = \
            natcap.invest.pygeoprocessing_0_3_3.geoprocessing.create_raster_from_vector_extents(cell_size,cell_size,\
            gdal.GDT_Float32, no_data, filename, gdal.OpenEx(aoi_uri))
        band = raster.GetRasterBand(1)
        array = band.ReadAsArray()
        # use raster to convert fetch_keys to point coordinates
        x, y = zip(*fetch_keys)
        coords = xy_to_rowcol(np.array(x), np.array(y), raster)
        coords = zip(coords[0], coords[1])
        array = np.ones_like(array) * no_data
        for p in range(len(coords)):
            i, j = coords[p]
            key = fetch_keys[p]
            array[i, j] = fetch[key][sector]

        # Save F(n) to disk
        band.WriteArray(array)
        raster_list.append(filename)

    return raster_list

# TODO: write a unit test for this function
# TODO: Update docstrings!
def save_fetch_depths(fetch, aoi_uri, cell_size, base_path, prefix):
    """ Create dictionary of raster filenames of fetch F(n) for each sector n.

        Inputs:
            - wind_data: wind data points adjusted to the aoi
            - aoi: used to create the rasters for each sector
            - cell_size: raster granularity in meters
            - base_path: base path where the generated raster will be saved

        Output:
            A dictionary where keys are sector angles in degrees and values are
            raster filenames where F(n) is defined on each cell"""
    no_data = -32768.
    field_prefix = os.path.join(base_path, prefix + 'fetch_depth')
    extension = '.tif'

    point_count = len(fetch.keys())
    # Save computed raster
    raster_list = []
    # Extract the # of sectors, assumed to start at angle 0 with identical span
    sector_count = 0
    fetch_keys = fetch.keys()
    for point in fetch_keys:
        sector_count = len(fetch[point])
        break
    # Build F(n), raster filename dictionary and save data to disk
    anomalies = {}
    for sector in range(sector_count):
        name = str(sector *360 /sector_count)
        # Open raster file
        filename = field_prefix +'_' +name +extension
        raster = \
            natcap.invest.pygeoprocessing_0_3_3.geoprocessing.create_raster_from_vector_extents(cell_size,cell_size,\
            gdal.GDT_Float32, no_data, filename, gdal.OpenEx(aoi_uri))
        band = raster.GetRasterBand(1)
        array = band.ReadAsArray()
        array = np.ones_like(array) * no_data
        # use raster to convert fetch_keys to point coordinates
        x, y = zip(*fetch_keys)
        coords = xy_to_rowcol(np.array(x), np.array(y), raster)
        coords = zip(coords[0], coords[1])
        # Build a fetch array F(n) for sector n
        for p in range(len(coords)):
            i, j = coords[p]
            key = fetch_keys[p]
            array[i, j] = fetch[key][sector]
        # Save F(n) to disk
        band.WriteArray(array)
        raster_list.append(filename)

    return raster_list

def save_array_to_raster(array, out_uri, base_uri, cell_size, \
    no_data = None, default_nodata = 0., gdal_type = gdal.GDT_Float32):
    """ Save an array to a raster constructed from an AOI.

        Inputs:
            - array: numpy array to be saved
            - out_uri: output raster file URI.
            - base_uri: URI to the AOI from which to construct the template raster
            - cell_size: granularity of the rasterization in meters
            - recompute_nodata: if True, recompute nodata to avoid
              interferece with existing raster data
            - no_data: value of nodata used in the function. If None,
              revert to default_nodata.
            - default_nodata: nodata used if no_data is set to none.
        Output:
            - save the array in a raster file constructed from the AOI of
              granularity specified by cell_size
            - Return the array uri."""
    message = \
        "In save_array_to_raster, " + base_uri + " is not a proper raster URI."
    raster = gdal.OpenEx(base_uri)
    assert raster is not None, message
    raster_array = raster.GetRasterBand(1).ReadAsArray()
    message = \
    'Error in save_array_to_raster: array shape ' + str(array.shape) + \
    ' disagrees with base raster shape ' + str(raster_array.shape)
    assert raster_array.shape == array.shape, message
    raster = None

    # Remove the file if it already exist
    if os.path.isfile(out_uri):
        os.remove(out_uri)

    if no_data == None:
        no_data = default_nodata
        if no_data is None:
            no_data = -9999

    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.new_raster_from_base_uri(base_uri, out_uri, 'GTiff', no_data,\
    gdal_type)
    raster = gdal.OpenEx(out_uri, gdal.GA_Update)
    raster.GetRasterBand(1).WriteArray(array)
    gdal.Dataset.__swig_destroy__(raster)
    raster = None
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.calculate_raster_stats_uri(out_uri)

    return out_uri

# TODO: write a unit test for this function
def raster_from_shapefile_uri(shapefile_uri, aoi_uri, cell_size, output_uri, \
    field=None, all_touched=False, nodata = 0., datatype = gdal.GDT_Float32):
    """Burn default or user-defined data from a shapefile on a raster.

        Inputs:
            - shapefile: the dataset to be discretized
            - aoi_uri: URI to an AOI shapefile
            - cell_size: coarseness of the discretization (in meters)
            - output_uri: uri where the raster will be saved
            - field: optional field name (string) where to extract the data
                from.
            - all_touched: optional boolean that indicates if we use GDAL's
              ALL_TOUCHED parameter when rasterizing.

        Output: A shapefile where:
            If field is specified, the field data is used as burn value.
            If field is not specified, then:
                - shapes on the first layer are encoded as 1s
                - the rest is encoded as 0"""
    shapefile = gdal.OpenEx(shapefile_uri)
    message = "Can't open shapefile " + shapefile_uri
    assert shapefile, message
    if aoi_uri == shapefile_uri:
        aoi = shapefile
    else:
        aoi = gdal.OpenEx(aoi_uri)
    # Create the raster that will contain the new data
    raster = \
        natcap.invest.pygeoprocessing_0_3_3.geoprocessing.create_raster_from_vector_extents(cell_size,
        cell_size, datatype, nodata, \
        output_uri, aoi)
    layer = shapefile.GetLayer(0)
    # Add the all_touched option
    options = ['ALL_TOUCHED='+str(all_touched).upper()]
    if field:
        # Burn the data in 'field' to a raster
        layer_id, _ = \
        get_layer_and_index_from_field_name(field, shapefile)
        layer = shapefile.GetLayer(layer_id)
        options = ['ATTRIBUTE='+field] + options
        gdal.RasterizeLayer(raster, [1], layer, options = options)
    else:
        gdal.RasterizeLayer(raster, [1], layer, burn_values = [1], \
            options = options)
    return output_uri

def cast_ray_fast(direction, d_max):
    """ March from the origin towards a direction until either land or a
    maximum distance is met.

        Inputs:
        - origin: algorithm's starting point -- has to be on sea
        - direction: marching direction
        - d_max: maximum distance to traverse
        - raster: land mass raster

        Returns the distance to the origin."""
    # Rescale the stepping vector so that its largest coordinate is 1
    unit_step = direction / np.fabs(direction).max()
    # Compute the length of the normalized vector
    unit_step_length = np.sqrt(np.sum(unit_step**2))
    # Compute the number of steps to take
    # Use ceiling to make sure to include any cell that is within the range of
    # max_fetch
    step_count = int(math.ceil(d_max / unit_step_length))
    I = np.array([i*unit_step[0] for i in range(step_count+1)])
    J = np.array([j*unit_step[1] for j in range(step_count+1)])

    return ((I, J), unit_step_length)

def fetch_vectors(angles):
    """convert the angles passed as arguments to raster vector directions.

        Input:
            -angles: list of angles in radians

        Outputs:
            -directions: vector directions numpy array of size (len(angles), 2)
    """
    # Raster convention: Up is north, i.e. decreasing 'i' is towards north.
    # Wind convention: Wind is defined as blowing FROM and not TOWARDS. This
    #                  means that fetch rays are going where the winds are
    #                  blowing from:
    # top angle: cartesian convention (x axis: J, y axis: negative I)
    # parentheses: (oceanographic
    #               convention)    Angle   direction   ray's I  ray's J
    #                                                  coord.   coord.
    #              90                  0      north       -1        0
    #             (90)                90       east        0        1
    #               |                180      south        1        0
    #               |                270       west        0       -1
    #     0         |         180
    #   (180)-------+-------->(0)  Cartesian to oceanographic
    #               |              angle transformation: a' = 180 - a
    #               |
    #               |              so that: [x, y] -> [I, J]
    #              270
    #             (270)
    #
    directions = np.empty((len(angles), 2))

    for a in range(len(angles)):
        pi = math.pi
        directions[a] = (round(math.cos(pi - angles[a]), 10),\
            round(math.sin(pi - angles[a]), 10))
    return directions

def compute_fetch_uri(landmass_raster_uri, rays_per_sector, d_max, cell_size, \
    shore_uri, bathymetry_uri):
    """ Given a land raster, return the fetch distance from a point
    in given directions

        - land_raster: raster where land is encoded as 1s, sea as 0s,
            and cells outside the area of interest as anything
            different from 0s or 1s.
        - directions: tuple of angles (in radians) from which the fetch
            will be computed for each pixel.
        - d_max: maximum distance in meters over which to compute the fetch
        - cell_size: size of a cell in meters
        - shore_uri: URI to the raster where the shoreline is encoded as 1s,
            the rest as 0s.

        returns: a tuple (distances, depths) where:
            distances is a dictionary of fetch data where the key is a shore
            point (tuple of integer coordinates), and the value is a 1*sectors
            numpy array containing fetch distances (float) from that point for
            each sector. The first sector (0) points eastward."""
    # Extract bathymetry information
    bathymetry_raster = gdal.OpenEx(bathymetry_uri)
    band = bathymetry_raster.GetRasterBand(1)
    bathymetry_nodata = band.GetNoDataValue()
    assert bathymetry_nodata is not None, \
        "Please, set nodata value for " + bathymetry_uri
    bathymetry = band.ReadAsArray()

    # Extract landmass information
    landmass_raster = gdal.OpenEx(landmass_raster_uri)
    land_array = landmass_raster.GetRasterBand(1).ReadAsArray()

    # Extract shore from raster
    shore_raster = gdal.OpenEx(shore_uri)
    shore_geo = shore_raster.GetGeoTransform()
    shore_array = shore_raster.GetRasterBand(1).ReadAsArray()
    message = 'shore and landmass raster sizes disagree:' + \
    str(shore_array.shape) + ' & ' + str(land_array.shape) + ' respectively.'
    assert shore_array.shape == land_array.shape, message
    shore_points = np.where(shore_array == 1)
    message = 'No shore points from which to compute the fetch.'
    assert shore_points[0].size > 0, message

    distance, avg_depth = compute_fetch(land_array, rays_per_sector, \
        d_max, cell_size, shore_points, bathymetry, bathymetry_nodata, \
        shore_geo, shore_raster)

    return (distance, avg_depth)

# TODO: Add docstring to this function
def compute_fetch(land_array, rays_per_sector, d_max, cell_size, \
    shore_points, bathymetry, bathymetry_nodata, GT, shore_raster):
    # precompute directions
    direction_count = SECTOR_COUNT * rays_per_sector
    direction_range = range(direction_count)
    direction_step = 2.0 * math.pi / direction_count
    directions_rad = [a * direction_step for a in direction_range]
    direction_vectors = fetch_vectors(directions_rad)
    unit_step_length = np.empty(direction_vectors.shape[0])
    # Perform a bunch of tests to ensure the assumptions in the fetch algorithm
    # are valid
    # Check that bathy and landmass rasters are size-compatible
    #print('land_shape', land_array.shape)
    #print('bathy_shape', bathymetry.shape)
    #print('i_max, j_max', np.amax(shore_points[0]), np.amax(shore_points[1]))
    #print('shore points', shore_points[0].size)
    message = 'landmass and bathymetry rasters are not the same size:' + \
    str(land_array.shape) + ' and ' + str(bathymetry.shape) + ' respectively.'
    assert land_array.shape == bathymetry.shape, message
    # Used to test if point fall within both land and bathy raster size limits
    (i_count, j_count) = land_array.shape
    # Check that shore points fall within the land raster limits
    message = 'some shore points fall outside the land raster'
    assert (np.amax(shore_points[0]) < i_count) and \
        (np.amax(shore_points[1]) < j_count), message
    # Check that shore points don't fall on nodata
    shore_points_on_nodata = np.where(land_array[shore_points] < 0.)[0].size
    message = 'There are ' + str(shore_points_on_nodata) + '/' + \
    str(shore_points[0].size) + \
    ' shore points on nodata areas in the land raster. There should be none.'
    assert shore_points_on_nodata == 0, message
    # Check that shore points don't fall on land
    shore_points_on_land = np.where(land_array[shore_points] > 0)[0].size
    if shore_points_on_land:
        points = np.where(land_array[shore_points] > 0)
        #print(points)
        points = (shore_points[0][points[0]], shore_points[1][points[0]])
        #print(points)
        #print(land_array[points])
        #print(zip(points[0], points[1]))
    message = 'There are ' + str(shore_points_on_land) + \
    ' shore points on land. There should be none.'
    assert shore_points_on_land == 0, message
    # Compute the ray paths in each direction to their full length (d_max).
    # We'll clip them for each point in two steps (raster boundaries & land)
    # The points returned by the cast function are relative to the origin (0,0)
    ray_path = {}
    valid_depths = 0 # used to determine if there are non-nodata depths
    for d in direction_range:
        result = \
        cast_ray_fast(direction_vectors[d], MAX_FETCH/cell_size)
        ray_path[directions_rad[d]] = result[0]
        unit_step_length[d] = result[1]
    # For each point, we use the rays in ray_path, and clip them in 2 steps:
    # 1)- clip the ray paths that go beyond the raster boundaries
    # 2)- If a ray passes over a landmass, remove that section till the end
    # All this computation has to be done on a per-point basis.
    point_list = np.array(zip(shore_points[0], shore_points[1]))
    #print('shore_geo', GT)
    x_points = shore_points[1] * GT[1] + GT[0] + shore_points[0] * GT[2]
    y_points = shore_points[1] * GT[4] + GT[3] + shore_points[0] * GT[5]
    x_points, y_points = \
    rowcol_to_xy(shore_points[0], shore_points[1], shore_raster)
    key_list = np.array(zip(x_points, y_points))
    distance = {}
    avg_depth = {}
    positive_depth_points = 0
    point_count = point_list.shape[0]
    for p in range(point_count):
        point = point_list[p]
        key = (key_list[p][0], key_list[p][1])
        #print(key)
        distance[key] = np.empty(direction_count)
        avg_depth[key] = np.empty(direction_count)
        for d in range(direction_count):
            direction = directions_rad[d]
            # Anchor the ray path to the current point
            I = ray_path[direction][0]+point[0]
            J = ray_path[direction][1]+point[1]
            # We need integer indices to index arrays: round I and J
            I = np.around(I).astype(int)
            J = np.around(J).astype(int)
            # Find valid indices for I and J separately
            valid_i = np.where((I>=0) & (I<i_count))
            valid_j = np.where((J>=0) & (J<j_count))
            # Find points for which both I and J are valid
            valid_i_and_j = set(valid_i[0]) & set(valid_j[0])
            # Put the indices back into a tuple as if it came from np.where()
            valid_i_and_j = (np.array([pt for pt in valid_i_and_j]),)
            # Isolate indices from points within the raster
            I = I[valid_i_and_j]
            J = J[valid_i_and_j]
            # At this point, all indices are within bounds
            # Extract only those ray indices that are not over land
            sea_indices = np.where(land_array[(I, J)] < 1)[0]
            # If not all indices over water -> keep first section over water
            if sea_indices[-1] != len(sea_indices) - 1:
                # Find the index after the first section over water:
                # Since ray indices over land don't show up in sea_indices, do
                # 1- Subtract each index from its predecessor (D_i=i_n - i_n-1)
                # 2- Continuous indices over water have a D_i == 1, otherwise
                #   the indices are separated by land (D_i > 1). We extract
                #   all indices that have a discontinuity, I=np.where(D_i > 1)
                non_consecutive = \
                    np.where(sea_indices[1:] - sea_indices[0:-1] > 1)[0]
                # 3- The index at I[0] is the end of the ray we want to keep
                #   (before the first time the ray hits land). We use I[0]+1
                #   for slicing ray.
                sea_indices = (sea_indices[:non_consecutive[0]+1],)
            # At this point, the ray stops before the first landmass
            # We use sea_indices to extract the accurate portion of the ray to
            # compute the distance
            I = I[sea_indices]
            J = J[sea_indices]
            # We now compute the distance traversed by the ray, which we almost
            # have already: we know the length of the ray when moving 1 cell
            # (by taking 1 step). The number of steps to get the ray is the
            # biggest of its coordinates:
            step_count = max(math.fabs(I[-1]-I[0]), math.fabs(J[-1]-J[0]))
            D = step_count * unit_step_length[d]
            # We want to return the maximum fetch distance: it's max_fetch if
            # the ray is not stopped by land before, else it's the maximum
            # distance the ray traversed over water. D is not this maximum
            # length: the marching algorithm makes 1 pixel jumps in either the
            # x or y direction starting at the center of the first pixel. So,
            # 1/2 of the last pixel is not accounted for in D. We have to take
            # this into account.
            to_last_pixel_edge = unit_step_length[d] / 2.
            # Fetch distance is distance from pixel center to edge of water
            distance[key][d] = min(MAX_FETCH, (D+to_last_pixel_edge)*cell_size)
            # reset invalid depth values from segments outside the bathymetry
            # layer
            depths = bathymetry[(I, J)]
            #print(bathymetry_nodata, depths.shape, depths)

            if depths[0] == bathymetry_nodata:
                average_depth = np.array([-100.0])
            # Valid depths, compute the average
            else:
                # Remove nodata from depth values
                valid_data = np.where(depths != bathymetry_nodata)
                if valid_data[0].size > 0:
                    depths = depths[valid_data]
                else:
                    print('Warning: point',point,'surrounded by nodata depths')
                # Remove positive values
                negative = np.where(depths <= 0.)[0]
                if negative.size > 0:
                    depths = depths[negative]
                # If depths are all positive, set to zero
                else:
                    depths = np.array([0.])
                # Average depth is mean bathymetry along the ray
                average_depth = np.average(depths)
                # Testing for Inf or NaN
                message = 'Detected NaN in average_depth.' + \
                    ' individual depths ' + str(depths) + \
                    ', valid depth indices: ' + str(negative)
                assert not np.isnan(average_depth), message
                message = 'Detected Inf in average_depth.' + \
                    ' individual depths ' + str(depths) + \
                    ', valid depth indices: ' + str(negative) + \
                    ', valid - no_data: ' + str(negative-bathymetry_nodata)
                assert not np.isinf(average_depth), message

            message = 'depth == nodata (' + str(bathymetry_nodata) + \
            '). bathymetry = ' + str(depths)
            assert average_depth != bathymetry_nodata, message
            avg_depth[key][d] = average_depth
        # We have the distances for all the directions, now we combine them
        # Shift the arrays so that each sector has an equal number of rays on
        # each side of its center
        distance[key] = np.roll(distance[key], (rays_per_sector / 2))
        avg_depth[key] = np.roll(avg_depth[key], (rays_per_sector / 2))
        # Reshape the fetch arrays so that a row corresponds to a sector
        distance[key] = \
            np.reshape(distance[key], (SECTOR_COUNT, rays_per_sector))
        avg_depth[key] = \
            np.reshape(avg_depth[key], (SECTOR_COUNT, rays_per_sector))
        # Compute the weights by taking the cos of the appropriately shifted
        # angles
        angles = np.array(directions_rad[:rays_per_sector])
        angles -= directions_rad[rays_per_sector / 2]
        cos = np.cos(angles)
        # Take the weighted rows average column-wise
        distance[key] = \
            np.minimum(np.average(distance[key] * cos, axis = 1), d_max)
        avg_depth[key] = np.average(avg_depth[key], axis = 1)
        pos_depth = np.where(avg_depth[key] >= 0)
        positive_depth_points += pos_depth[0].size

    if pos_depth[0].size:
        message = str(pos_depth[0].size) + \
        ' points have positive depth, set to -1.'
        LOGGER.warning(message)
        avg_depth[key][pos_depth] = -1

    return (distance, avg_depth)


def adjust_raster_to_aoi(in_dataset_uri, aoi_datasource_uri, cell_size, \
    out_dataset_uri):
    """Adjust in_dataset_uri to match aoi_dataset_uri's extents, cell size and
    projection.

        Inputs:
            - in_dataset_uri: the uri of the dataset to adjust
            - aoi_dataset_uri: uri to the aoi we want to use to adjust
                in_dataset_uri
            - out_dataset_uri: uri to the adjusted dataset

        Returns:
            - out_dataset_uri"""
    # Split the path apart from the filename
    out_head, out_tail = os.path.split(out_dataset_uri)
    _, tail = os.path.split(in_dataset_uri)
    # Split the file basename from the file extension
    out_base, _ = os.path.splitext(out_tail)
    base, _ = os.path.splitext(tail)
    # Preliminary variable initialization
    aoi_wkt = shapefile_wkt(gdal.OpenEx(aoi_datasource_uri))
    input_wkt = raster_wkt(gdal.OpenEx(in_dataset_uri))
    message = "Cannot extract a Well Known Transformation (wkt) from " + \
        str(in_dataset_uri) + ". Please check the file has a valid WKT."
    assert input_wkt, message
    # Reproject AOI to input dataset projection
    reprojected_aoi_uri = os.path.join(out_head, base + '_reprojected_aoi.shp')
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.reproject_datasource_uri(aoi_datasource_uri, input_wkt, \
    reprojected_aoi_uri)
    # Clip dataset with reprojected AOI
    clipped_dataset_uri = os.path.join(out_head, out_base + '_unprojected.tif')
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.clip_dataset_uri(in_dataset_uri, reprojected_aoi_uri, \
    clipped_dataset_uri, False)
    # Reproject clipped dataset to AOI's projection
    #natcap.invest.pygeoprocessing_0_3_3.geoprocessing.reproject_dataset_uri(clipped_dataset_uri, cell_size, \
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.reproject_dataset_uri(clipped_dataset_uri, \
    cell_size, aoi_wkt, 'bilinear', out_dataset_uri)
    # Done, return the dataset uri
    return out_dataset_uri

# TODO: improve this docstring!
def detect_shore_uri(landmass_raster_uri, aoi_raster_uri, output_uri):
    """ Extract the boundary between land and sea from a raster.

        - raster: numpy array with sea, land and nodata values.

        returns a numpy array the same size as the input raster with the shore
        encoded as ones, and zeros everywhere else."""
    landmass_raster = gdal.OpenEx(landmass_raster_uri)
    land_sea_array = landmass_raster.GetRasterBand(1).ReadAsArray()
    landmass_raster = None
    aoi_raster = gdal.OpenEx(aoi_raster_uri)
    aoi_array = aoi_raster.GetRasterBand(1).ReadAsArray()
    aoi_nodata = aoi_raster.GetRasterBand(1).GetNoDataValue()
    aoi_raster = None

    shore_array = detect_shore(land_sea_array, aoi_array, aoi_nodata)

    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.new_raster_from_base_uri( \
        aoi_raster_uri, output_uri, 'GTiff', 0., gdal.GDT_Float32)
    raster = gdal.OpenEx(output_uri, gdal.GA_Update)
    band = raster.GetRasterBand(1)
    band.FlushCache()
    band.WriteArray(shore_array)

# TODO: improve this docstring!
def detect_shore(land_sea_array, aoi_array, aoi_nodata):
    """ Extract the boundary between land and sea from a raster.

        - raster: numpy array with sea, land and nodata values.

        returns a numpy array the same size as the input raster with the shore
        encoded as ones, and zeros everywhere else."""
    # Rich's super-short solution, which uses convolution.
    nodata = -1
    land_sea_array[aoi_array == aoi_nodata] = nodata
    # Don't bother computing anything if there is only land or only sea
    land_size = np.where(land_sea_array > 0)[0].size

    if land_size == 0:
        LOGGER.warning('There is no shore to detect: land area = 0')
        return np.zeros_like(land_sea_array)
    elif land_size == land_sea_array.size:
        LOGGER.warning('There is no shore to detect: sea area = 0')
        return np.zeros_like(land_sea_array)
    else:
        kernel = np.array([[ 0, -1,  0],
                           [-1,  4, -1],
                           [ 0, -1,  0]])
        # Generate the nodata shore artifacts
        aoi_array = np.ones_like(land_sea_array)
        aoi_array[land_sea_array == nodata] = nodata
        aoi_borders = (sp.signal.convolve2d(aoi_array, \
                                                kernel, \
                                                mode='same') <0 ).astype('int')
        # Generate all the borders (including data artifacts)
        borders = (sp.signal.convolve2d(land_sea_array, \
                                     kernel, \
                                     mode='same') <0 ).astype('int')
        # Real shore = all borders - shore artifacts
        borders = ((borders - aoi_borders) >0 ).astype('int') * 1.

        shore_segment_count = np.sum(borders)
        if shore_segment_count == 0:
            LOGGER.warning('No shore segment detected')
        return borders

def preprocess_dataset(dataset_uri, aoi_uri, cell_size, output_uri):
    """Funstion that preprocesses an input dataset (clip,
    reproject, resample) so that it is ready to be used in the model

        Inputs:
            -dataset_uri: uri to the input dataset to be pre-processed
            -aoi_uri: uri to an aoi polygon datasource that is used for
                clipping and reprojection.
            -cell_size: output dataset cell size in meters (integer)
            -output_uri: uri to the pre-processed output dataset.

        Returns output_uri
    """
    # Adjust the dataset to the aoi and save the result
    adjust_raster_to_aoi(dataset_uri, aoi_uri, cell_size, output_uri)

    return output_uri

def is_point_datasource(uri):
    """Returns True if the datasource is a point shapefile

        Inputs:
            -uri: uri to a datasource

        Outputs:
            -True if uri points to a point datasource, False otherwise
    """
    shapefile = gdal.OpenEx(uri)

    if shapefile is None:
        return False

    layer = shapefile.GetLayer(0)
    feature = layer.GetFeature(0)
    feature_definition = feature.GetDefnRef()
    geometry_type = feature_definition.GetGeomType()

    return geometry_type == ogr.wkbPoint

def is_polygon_datasource(uri):
    """Returns True if the datasource is a polygon shapefile

        Inputs:
            -uri: uri to a datasource

        Outputs:
            -True if uri points to a polygon datasource, False otherwise
    """
    shapefile = gdal.OpenEx(uri)

    if shapefile is None:
        return False

    layer = shapefile.GetLayer(0)
    feature = layer.GetFeature(0)
    feature_definition = feature.GetDefnRef()
    geometry_type = feature_definition.GetGeomType()

    return geometry_type == ogr.wkbPolygon

def preprocess_point_datasource(datasource_uri, aoi_uri, cell_size, \
    output_uri, field_list, nodata=0.):
    """Function that converts a point shapefile to a dataset by clipping,
    reprojecting, resampling, burning, and extrapolating burnt values.

        Inputs:
            -datasource_uri: uri to the datasource to be pre-processed
            -aoi_uri: uri to an aoi polygon datasource that is used for
                clipping and reprojection.
            -cell_size: output dataset cell size in meters (integer)
            -output_uri: uri to the pre-processed output dataset.
            -field_name: name of the field in the attribute table to get the
            values from. If a number, use it as a constant. If Null, use 1.

        Returns output_uri"""
    # Split the path apart from the filename
    head, tail = os.path.split(output_uri)
    # Split the file basename from the file extension
    base, ext = os.path.splitext(tail)
    # Insert a suffix to the file basename and glue the new path together
    aoi_adjusted_uri = os.path.join(head, base + '_adjusted' + ext)
    # Adjust the shapefile to the aoi and save the result
    adjust_shapefile_to_aoi(datasource_uri, aoi_uri, aoi_adjusted_uri)
    # Burn the field onto a raster
    raster_uri_list = []
    # Get the datasource field
    #print('field_list', field_list)
    for field_name in field_list:
        #print('field_name', field_name)
        if has_field(field_name, gdal.OpenEx(datasource_uri), \
            case_sensitive = False):
            # Create unique raster name
            raster_uri_list.append(os.path.join(head, base + '_' + \
            field_name + ext))
            # Create empty raster
            natcap.invest.pygeoprocessing_0_3_3.geoprocessing.create_raster_from_vector_extents(cell_size, \
            cell_size, gdal.GDT_Float32, nodata, raster_uri_list[-1], \
            gdal.OpenEx(aoi_uri))
            # Vectorize data
            natcap.invest.pygeoprocessing_0_3_3.geoprocessing.vectorize_points_uri(aoi_adjusted_uri, field_name, \
            raster_uri_list[-1])
        else:
            # field doesn't exist in data
            message = 'field ' + str(field_name) + " doesn't exist in " + \
            datasource_uri
            assert len(raster_uri_list)>0, message

    return raster_uri_list

def preprocess_polygon_datasource(datasource_uri, aoi_uri, cell_size, \
    output_uri, field_name = None, all_touched = False, nodata = 0., \
    empty_raster_allowed = False):
    """Function that converts a polygon shapefile to a dataset by clipping,
    reprojecting, resampling, burning, and extrapolating burnt values.

        Inputs:
            -datasource_uri: uri to the datasource to be pre-processed
            -aoi_uri: uri to an aoi polygon datasource that is used for
                clipping and reprojection.
            -cell_size: output dataset cell size in meters (integer)
            -output_uri: uri to the pre-processed output dataset.
            -field_name: name of the field in the attribute table to get the
            values from. If a number, use it as a constant. If Null, use 1.
            -all_touched: boolean flag used in gdal's vectorize_rasters
            options flag
            -nodata: float used as nodata in the output raster
            -empty_raster_allowed: flag that allows the function to return an
            empty raster if set to True, or break if set to False. False is the
            default.

        Returns output_uri"""
    # Split the path apart from the filename
    head, tail = os.path.split(output_uri)
    # Split the file basename from the file extension
    base, _ = os.path.splitext(tail)
    # Insert a suffix to the file basename and glue the new path together
    aoi_adjusted_uri = os.path.join(head, base + '_aoi_adjusted')
    # Adjust the shapefile to the aoi and save the result
    aoi_adjusted_uri = adjust_shapefile_to_aoi(
        datasource_uri, aoi_uri, aoi_adjusted_uri, empty_raster_allowed)
    # Burn the adjusted file to a raster
    raster_from_shapefile_uri(aoi_adjusted_uri, aoi_uri, cell_size, \
        output_uri, field_name, all_touched, nodata)

    return output_uri

def find_attribute_field(field_name, shapefile_uri):
    """Look for a field name in the shapefile attribute table. Search is case
    insensitive."""
    # Look through all the fileds in the attribute table
    shapefile_fields = enumerate_shapefile_fields(shapefile_uri)
    for shapefile_field in shapefile_fields[0]:
        # Do a case-insensitive search
        if shapefile_field.lower() == field_name.lower():
            return shapefile_field
    # If we're here, the field is not in the attribute table, print a useful
    # help message and assert
    LOGGER.debug('available fileds in the shapefile:')
    for shapefile_field in shapefile_fields:
        LOGGER.debug(shapefile_field)
    message = 'File ' + shapefile_uri + " doesn't have field " + field_name
    assert False, message

#:RICH: move this stuff to the 'UI' layer
def preprocess_inputs(args):
    #print('---------------------------------------------------------')
    #print('args:')
    #for item in args.items():
    #    print(item)
    #print('---------------------------------------------------------')
    # Convert anything in unicode to utf-8:
    for key in args.keys():
        if type(args[key]) is unicode:
            args[key] = args[key].encode('utf-8')

    args["uri"] = {}

    # Set the zero-oceanic wave power threshold to max_fetch
    args['H_threshold'] = args['max_fetch']

    # Adjust the maximum fetch to a value that is as small as possible
    MAX_FETCH = args['max_fetch']
    # Warn the user if the fetch is too large, that it will take more time to
    # compute
    if MAX_FETCH > 60000:
        LOGGER.warning('Maximum fetch value is set beyond 60km:' + \
        ' Please, pay extra attention to the interpretation' + \
        ' of the vulnerability of sheltered segments which fetch' + \
        ' is beyond 60km.')

    args['H_threshold'] = args['max_fetch']
    prefix = '00_PRE_'
    suffix = '' if 'suffix' not in args else '_' + args['suffix']
    args['intermediate_directory'] = \
        os.path.join(args['workspace_dir'], 'intermediate')
    args['outputs_directory'] = \
        os.path.join(args['workspace_dir'], 'outputs')
    subdirectory = \
        os.path.join(args['intermediate_directory'], '00_preprocessing')
    # Create intermediate directories
    if not os.path.isdir(args['intermediate_directory']):
        os.makedirs(args['intermediate_directory'])
    if not os.path.isdir(subdirectory):
        os.makedirs(subdirectory)
    if not os.path.isdir(args['outputs_directory']):
        os.makedirs(args['outputs_directory'])

    # Args keys of all the URIs generated in this function used for
    # post-processing
    generated_uris = []

    args['aoi_raster_uri'] = \
        preprocess_polygon_datasource(args['aoi_uri'], args['aoi_uri'], \
        args['cell_size'], os.path.join(subdirectory, prefix + 'aoi.tif'))
    generated_uris.append(args['aoi_raster_uri'])

    # Set of input URIs that have been preprocessed
    preprocessed_uris = set()
    # URIs produced from a given input URI, organized as a dictionary where the
    # values are the unprocessed uris, and the keys are the corresponding
    # output uris.
    # output_uris[args['uri']] -> output_uri
    output_uris = {}
    # Do we need an output for this layer?
    if 'structures_uri' in args:
        # Is the input URI already preprocessed?
        if args['structures_uri'] in preprocessed_uris:
            # Yes: pull the output URI from the dictionary output_uris
            args['structures_raster_uri'] = output_uris[args['structures_uri']]
        else:
            # No, compute the output, and save the URI in output_uris
            preprocessed_uris.add(args['structures_uri'])
            print('Pre-processing structures...')
            args['structures_raster_uri'] = \
            preprocess_polygon_datasource(args['structures_uri'], \
            args['aoi_uri'], args['cell_size'], \
            os.path.join(subdirectory, prefix +'structures.tif'))
            output_uris[args['structures_uri']] = args['structures_raster_uri']
        generated_uris.append(args['structures_raster_uri'])
    # Do we need an output for this layer?
    if 'landmass_uri' in args:
        # Is the input URI already preprocessed?
        if args['landmass_uri'] in preprocessed_uris:
            # Yes: pull the output URI from the dictionary output_uris
            args['landmass_raster_uri'] = output_uris[args['landmass_uri']]
        else:
            # No, compute the output, and save the URI in output_uris
            preprocessed_uris.add(args['landmass_uri'])
            print('Pre-processing landmass...')
            args['landmass_raster_uri'] = \
            preprocess_polygon_datasource(args['landmass_uri'], \
            args['aoi_uri'], args['cell_size'], \
            os.path.join(subdirectory, prefix + 'landmass.tif'))
            output_uris[args['landmass_uri']] = args['landmass_raster_uri']
        generated_uris.append(args['landmass_raster_uri'])
    # Can we compute the shore?
    if ('aoi_raster_uri' in args) and ('landmass_raster_uri' in args):
        print('detecting shore...')
        args['shore_raster_uri'] = os.path.join(\
            subdirectory, prefix + 'shore.tif')
        detect_shore_uri( \
            args['landmass_raster_uri'], args['aoi_raster_uri'], \
            args['shore_raster_uri'])
        generated_uris.append(args['shore_raster_uri'])
    # Do we need an output for this layer?
    if 'bathymetry_uri' in args:
        # Is the input URI already preprocessed?
        if args['bathymetry_uri'] in preprocessed_uris:
            # Yes: pull the output URI from the dictionary output_uris
            args['bathymetry_raster_uri'] = output_uris[args['bathymetry_uri']]
        else:
            # No, compute the output, and save the URI in output_uris
            preprocessed_uris.add(args['bathymetry_uri'])
            print('Pre-processing bathymetry...')
            args['bathymetry_raster_uri'] = \
            preprocess_dataset(args['bathymetry_uri'], \
            args['aoi_uri'], args['cell_size'], \
            os.path.join(subdirectory, prefix + 'bathymetry.tif'))
            output_uris[args['bathymetry_uri']] = args['bathymetry_raster_uri']
            # Adjust bathymetry w.r.t. the mean sea level datum:
            if args['mean_sea_level_datum'] != 0:
                raster = \
                gdal.OpenEx(args['bathymetry_raster_uri'], gdal.GA_Update)
                band = raster.GetRasterBand(1)
                array = band.ReadAsArray()
                array -= args['mean_sea_level_datum']
                band.WriteArray(array)
        generated_uris.append(args['bathymetry_raster_uri'])
    if 'population_uri' in args:
        # Is the input URI already preprocessed?
        if args['population_uri'] in preprocessed_uris:
            # Yes: pull the output URI from the dictionary output_uris
            args['population_raster_uri'] = output_uris[args['population_uri']]
        else:
            # No, compute the output, and save the URI in output_uris
            preprocessed_uris.add(args['population_uri'])
            print('Pre-processing population...')
            args['population_raster_uri'] = \
            preprocess_dataset(args['population_uri'], \
            args['aoi_uri'], args['cell_size'], \
            os.path.join(subdirectory, prefix + 'population.tif'))
            output_uris[args['population_uri']] = args['population_raster_uri']
        generated_uris.append(args['population_raster_uri'])
    # Can we compute the fetch?
    if ('landmass_raster_uri' in args) and ('bathymetry_raster_uri') \
        and ('shore_raster_ri'):
        print('computing fetch...')
        # Uniformize the size of shore, land, and bathymetry rasters
        in_raster_list = [args['landmass_raster_uri'], \
            args['shore_raster_uri'], args['bathymetry_raster_uri']]

        tmp_landmass_raster = natcap.invest.pygeoprocessing_0_3_3.geoprocessing.temporary_filename()
        (head, tail) = os.path.split(args['shore_raster_uri'])
        tmp_bathy_raster = natcap.invest.pygeoprocessing_0_3_3.geoprocessing.temporary_filename()
        tmp_raster_uri = natcap.invest.pygeoprocessing_0_3_3.geoprocessing.temporary_filename()

        out_raster_list = [tmp_landmass_raster, tmp_raster_uri,\
            tmp_bathy_raster]

        cell_size = \
        natcap.invest.pygeoprocessing_0_3_3.geoprocessing.get_cell_size_from_uri(args['landmass_raster_uri'])
        resample_method_list = ['bilinear'] * len(out_raster_list)
        out_pixel_size = cell_size
        mode = 'dataset'
        dataset_to_align_index = 0
        dataset_to_bound_index = 0

        natcap.invest.pygeoprocessing_0_3_3.geoprocessing.align_dataset_list( \
        in_raster_list, out_raster_list, resample_method_list,
        out_pixel_size, mode, dataset_to_align_index, dataset_to_bound_index)

        shutil.copy(tmp_landmass_raster, args['landmass_raster_uri'])
        shutil.copy(tmp_bathy_raster, args['bathymetry_raster_uri'])

        landmass_raster_shape = \
            natcap.invest.pygeoprocessing_0_3_3.geoprocessing.get_row_col_from_uri(args['landmass_raster_uri'])
        shore_raster_shape = \
            natcap.invest.pygeoprocessing_0_3_3.geoprocessing.get_row_col_from_uri(args['shore_raster_uri'])
        bathymetry_raster_shape = natcap.invest.pygeoprocessing_0_3_3.geoprocessing.get_row_col_from_uri( \
            args['bathymetry_raster_uri'])

        assert (landmass_raster_shape == shore_raster_shape) and \
            (landmass_raster_shape == bathymetry_raster_shape)

        args['fetch_distances'], args['fetch_depths'] = \
             compute_fetch_uri(args['landmass_raster_uri'], \
             args['rays_per_sector'], MAX_FETCH, \
             args['cell_size'], args['shore_raster_uri'], \
             args['bathymetry_raster_uri'])
        array = None
        bathymetry = None
        # The shore segments might have been removed from shore_array.
        # Store the new shore in the raster.

        # Making sure the fetch depth is correct
        for point in args['fetch_distances'].keys():
            value = args['fetch_distances'][point]
            message = 'Negative fetch distance:' + str(value) + ' for point ' + str(point)
            assert (value >= 0).all(), message
        for point in args['fetch_depths'].keys():
            value = args['fetch_depths'][point]
            message = 'Fetch depth is ' + str(value) + ' for point ' + str(point)
            assert (value >= -20000).all(), message

        args['fetch_distance_uris'] = \
            save_fetch_distances(args['fetch_distances'], args['aoi_uri'], \
            args['cell_size'], subdirectory, prefix)
        args['fetch_depth_uris'] = \
            save_fetch_depths(args['fetch_depths'], args['aoi_uri'], \
            args['cell_size'], subdirectory, prefix)
        print('done.')
    else:
        print('Missing bathymetry, skipping the computation of fetch.')
    # Do we need an output for this layer?
    if 'geomorphology_uri' in args:
        #
        # Is the input URI already preprocessed?
        if args['geomorphology_uri'] in preprocessed_uris:
            # Yes: pull the output URI from the dictionary output_uris
            args['geomorphology_raster_uri'] = \
            output_uris[args['geomorphology_uri']]
        else:
            # No, compute the output, and save the URI in output_uris
            preprocessed_uris.add(args['geomorphology_uri'])
            print('Pre-processing geomorphology...')
            args['geomorphology_raster_uri'] = \
            preprocess_polygon_datasource(
                args['geomorphology_uri'], args['aoi_uri'], args['cell_size'],
                os.path.join(subdirectory, prefix + 'geomorphology.tif'),
                field_name=find_attribute_field('RANK', args['geomorphology_uri']),
                all_touched=True, nodata = 0.0)
            output_uris[args['geomorphology_uri']] = \
            args['geomorphology_raster_uri']
        generated_uris.append(args['geomorphology_raster_uri'])
    # Do we need an output for this layer?
    if 'relief_uri' in args:
        # Is the input URI already preprocessed?
        if args['relief_uri'] in preprocessed_uris:
            # Yes: pull the output URI from the dictionary output_uris
            args['relief_raster_uri'] = output_uris[args['relief_uri']]
        else:
            # No, compute the output, and save the URI in output_uris
            preprocessed_uris.add(args['relief_uri'])
            print('Pre-processing relief...')
            args['relief_raster_uri'] = \
            preprocess_dataset(args['relief_uri'], \
            args['aoi_uri'], args['cell_size'], \
            os.path.join(subdirectory, prefix + 'relief.tif'))
            #preprocess_polygon_datasource(args['relief_uri'], \
            #args['aoi_uri'], args['cell_size'], \
            #os.path.join(subdirectory, prefix + 'relief.tif'), \
            #field_name = find_attribute_field('RANK', args['relief_uri']), \
            #all_touched = True, nodata = 0.)
            output_uris[args['relief_uri']] = args['relief_raster_uri']
        generated_uris.append(args['relief_raster_uri'])
    if 'habitats_csv_uri' in args:
        assert os.path.exists(args['habitats_directory_uri']), \
        'Missing directory.'
        # Predefined constants
        NAME                = 0 # Habitat name (string)
        FILE_NAME           = 4 # Name of the habitat shapefile
        ATTRIBUTE_COUNT     = 5 # Total number of attributes
        # Dictionary that contains the data's uri indexed by name

        args['habitat_data'] = read_habitat_info(args['habitats_csv_uri'], \
        args['habitats_directory_uri'])

        # Process each habitat
        for habitat in args['habitat_data']:
            habitat_data = args['habitat_data'][habitat]
            # Check if there is a habitat layer to work with:
            habitat_key = habitat_data[NAME].replace(' ', '_')
            if len(habitat_data) != ATTRIBUTE_COUNT:
                LOGGER.warning( \
                'No shapefile data associated with habitat %s. Skipping.' % \
                habitat_data[NAME])
            else:
                print('Pre-processing habitat ' + str(habitat_data[NAME]) + \
                    '...')
                args[habitat_key] = os.path.join(subdirectory, \
                prefix + habitat_key + '.tif')
                #print('processing ' + \
                #    os.path.basename(habitat_data[FILE_NAME]))
                preprocess_polygon_datasource(habitat_data[FILE_NAME], \
                args['aoi_uri'], args['cell_size'], args[habitat_key], \
                all_touched = True, nodata = 0., empty_raster_allowed = True)
                # Drop the habitat file if it's empty (no features in the AOI)
                raster = gdal.OpenEx(args[habitat_key])
                habitat_array = raster.GetRasterBand(1).ReadAsArray()
                captured_habitat = np.where(habitat_array > 0.)[0].size > 0
                if captured_habitat:
                    habitat_data.append(args[habitat_key])
                else:
                    print('Habitat ' + str(habitat_data[NAME]) + \
                        ' is not in the AOI, and has been Skipped.')
                    habitat_data.append('skip')
            generated_uris.append(args[habitat_key])
    if ('climatic_forcing_uri' in args) and ('bathymetry_uri' in args) and \
        ('fetch_distances' in args) and ('fetch_depths' in args):
        print('Pre-processing climatic forcing...')
        field_names = enumerate_shapefile_fields(args['climatic_forcing_uri'])
        REI_V_names = []
        REI_PCT_names = []
        WavP_names = []
        WavPPCT_names = []
        for field_name in field_names[0]:
            if re.match('REI_V', field_name):
                REI_V_names.append(field_name)
            elif re.match('REI_PCT', field_name):
                REI_PCT_names.append(field_name)
            elif re.match('WavP_', field_name):
                WavP_names.append(field_name)
            elif re.match('WavPPCT', field_name):
                WavPPCT_names.append(field_name)
        #print('Vectorizing data from REI_V...')
        args['REI_V'] = \
        preprocess_point_datasource(args['climatic_forcing_uri'], \
        args['aoi_uri'], args['cell_size'], \
        os.path.join(subdirectory, prefix + 'climatic_forcing.tif'), \
        field_list = REI_V_names, nodata = 0.)
        generated_uris += args['REI_V']
        #print('Vectorizing data from REI_PCT...')
        args['REI_PCT'] = \
        preprocess_point_datasource(args['climatic_forcing_uri'], \
        args['aoi_uri'], args['cell_size'], \
        os.path.join(subdirectory, prefix + 'climatic_forcing.tif'), \
        field_list = REI_PCT_names, nodata = 0.)
        generated_uris += args['REI_PCT']
        #print('Vectorizing data from WavP_...')
        args['WavP'] = \
        preprocess_point_datasource(args['climatic_forcing_uri'], \
        args['aoi_uri'], args['cell_size'], \
        os.path.join(subdirectory, prefix + 'climatic_forcing.tif'), \
        field_list = WavP_names, nodata = 0.)
        generated_uris += args['WavP']
        #print('Vectorizing data from WavPPCT...')
        args['WavPPCT'] = \
        preprocess_point_datasource(args['climatic_forcing_uri'], \
        args['aoi_uri'], args['cell_size'], \
        os.path.join(subdirectory, prefix + 'climatic_forcing.tif'), \
        field_list = WavPPCT_names, nodata = 0.)
        generated_uris += args['WavPPCT']
    if 'continental_shelf_uri' in args:
        # Is the input URI already preprocessed?
        if args['continental_shelf_uri'] in preprocessed_uris:
            # Yes: pull the output URI from the dictionary output_uris
            args['continental_shelf_raster_uri'] = \
            output_uris[args['continental_shelf_uri']]
        else:
            # No, compute the output, and save the URI in output_uris
            preprocessed_uris.add(args['continental_shelf_uri'])
            print('Pre-processing continental shelf...')
            args['continental_shelf_raster_uri'] = \
            preprocess_polygon_datasource(args['continental_shelf_uri'], \
            args['aoi_uri'], args['cell_size'], \
            os.path.join(subdirectory, prefix + 'continental_shelf.tif'), \
            nodata = 0.)
            output_uris[args['continental_shelf_uri']] = \
            args['continental_shelf_raster_uri']
            # Ideally, we'd want the continental shelf to be filled with 1s,
            # and the rest with 0s. However, the polygons in the data is not
            # the continental shelf itself, so the 1s are not what we want.
            # But we can flip all 1s to 0s, and all 0s to 1s (logical_not) and
            # we get the contiental shelf that we want, that ends at the right
            # place. We also have another band offshore that is encoded as the
            # continental shelf, but that's not. THis is not a problem, because
            # this portion of 1s is isolated from the real shelf by a band of
            # 0s, which is good enough for the distance measure we need to
            # compute later on.
            cont_shelf_raster = \
            gdal.OpenEx(args['continental_shelf_raster_uri'], gdal.GA_Update)
            cont_shelf_array = cont_shelf_raster.GetRasterBand(1).ReadAsArray()
            cont_shelf_raster.GetRasterBand(1).WriteArray(np.logical_not( \
            cont_shelf_array).astype(float))
            cont_shelf_raster = None
        generated_uris.append(args['continental_shelf_raster_uri'])
    if 'sea_level_rise_uri' in args:
        # Is the input URI already preprocessed?
        if args['sea_level_rise_uri'] in preprocessed_uris:
            # Yes: pull the output URI from the dictionary output_uris
            args['sea_level_rise_raster_uri'] = \
            output_uris[args['sea_level_rise_raster_uri']]
        else:
            # No, compute the output, and save the URI in output_uris
            preprocessed_uris.add(args['sea_level_rise_uri'])
            print('Pre-processing sea level rise...')
            if(is_point_datasource(args['sea_level_rise_uri'])):
                field = get_field( \
                    'Trend', \
                    gdal.OpenEx(args['sea_level_rise_uri']), \
                    case_sensitive = False)
                # Can't find field, assert after printing a useful error message
                if field is None:
                    message = 'Layer ' + args['sea_level_rise_uri'] + \
                    " doesn't have the field " + field
                    LOGGER.error(message)
                    fields = enumerate_shapefile_fields(args['sea_level_rise_uri'])
                    LOGGER.debug('Available fields: ' + str(len(fields)))
                    # Show all ther fields in the shapefile
                    for f in fields:
                        LOGGER.debug(f)
                    message = 'field ' + field + ' cannot be found in ' + \
                        str(args['sea_level_rise_raster_uri'])
                    assert field is not None, message
                args['sea_level_rise_raster_uri'] = \
                preprocess_point_datasource(args['sea_level_rise_uri'],\
                args['aoi_uri'], args['cell_size'], \
                os.path.join(subdirectory, prefix + 'sea_level_rise.tif'), \
                field_list = [field], nodata = 0.)[0]
            if(is_polygon_datasource(args['sea_level_rise_uri'])):
                field = get_field( \
                    'Trend', \
                    gdal.OpenEx(args['sea_level_rise_uri']), \
                    case_sensitive = False)
                if field is None:
                    message = 'Layer ' + args['sea_level_rise_uri'] + \
                    " doesn't have the field Trend"
                    LOGGER.error(message)
                    fields = enumerate_shapefile_fields(args['sea_level_rise_uri'])
                    LOGGER.debug('Available fields: ' + str(len(fields)))
                    # Show all ther fields in the shapefile
                    for f in fields:
                        LOGGER.debug(f)
                # Preprocesses sea_level_rise_raster_uri
                args['sea_level_rise_raster_uri'] = \
                preprocess_polygon_datasource(args['sea_level_rise_uri'], \
                args['aoi_uri'], args['cell_size'], \
                os.path.join(subdirectory, prefix + 'sea_level_rise.tif'), \
                field_name = field, nodata = 0.)
            output_uris[args['sea_level_rise_uri']] = \
            args['sea_level_rise_raster_uri']
        generated_uris.append(args['sea_level_rise_raster_uri'])
    if 'additional_layer_uri' in args:
        # Is the input URI already preprocessed?
        if args['additional_layer_uri'] in preprocessed_uris:
            # Yes: pull the output URI from the dictionary output_uris
            args['additional_layer_raster_uri'] = \
            output_uris[args['additional_layer_uri']]
        else:
            # No, compute the output, and save the URI in output_uris
            preprocessed_uris.add(args['additional_layer_uri'])
            print('pre-processing additional layer')
            args['additional_layer_raster_uri'] = \
            preprocess_point_datasource(args['additional_layer_uri'],\
            args['aoi_uri'], args['cell_size'], \
            os.path.join(subdirectory, prefix + 'additional_layer.tif'), \
            field_list = [field], nodata = 0.)[0]
            output_uris[args['additional_layer_uri']] = \
            args['additional_layer_raster_uri']
        generated_uris.append(args['additional_layer_raster_uri'])

    # Resizing and aligning datasets so that the size of their arrays are
    # mutually compatible.
    # Temporary URIs to avoid data collision
    temporary_uris = \
    [natcap.invest.pygeoprocessing_0_3_3.geoprocessing.temporary_filename() for i in range(len(generated_uris))]
    # Align/resize the rasters and store them in temporary uris
    print('generated_uris')
    natcap.invest.pygeoprocessing_0_3_3.geoprocessing.align_dataset_list(generated_uris, temporary_uris, \
    ["bilinear"] * len(generated_uris), args['cell_size'], \
    'intersection', 0)
    # Copy everything back into place
    for uri in range(len(generated_uris)):
        shutil.copy(temporary_uris[uri], generated_uris[uri])
        raster = gdal.OpenEx(generated_uris[uri])
        raster_shape = raster.GetRasterBand(1).ReadAsArray().shape
        raster = None
        #print(raster_shape, os.path.basename(generated_uris[uri]))
    return args

def read_habitat_info(habitats_csv_uri, habitats_directory_uri):
    """Extract the habitats information from the csv file and directory."""
    extension = '.shp'      # Shapefile extension to look for
    ID = 1 # Habitat ID (unique int identifier)
    # Read the CSV file
    csv_reader = csv.reader(open(habitats_csv_uri))
    csv_data = {}
    csv_reader.next()
    for item in csv_reader:
        missing_elements = 4 - len(item)
        message = \
            'Missing ' + str(missing_elements) + ' element(s) from line ' + \
             str(item) + ' in ' + habitats_csv_uri
        assert missing_elements <= 0, message
        csv_data[item[ID].strip()] = item
    message = "Couldn't extract any data from the CSV file " + habitats_csv_uri
    assert csv_data, message
    # Retreive shapefiles by extension
    file_list = os.listdir(habitats_directory_uri)
    for entry in file_list:
        # Look for files with '.shp' extension...
        # ...AND '.shp' only: skip extentions like '.shp.xml'
        if extension in entry and \
            (len(entry) == entry.find(extension)+len(extension)):
            # Extract the basename, so that we can find the habitat ID
            basename = entry[0:entry.find(extension)]
            # Find the habitat ID for this file
            habitat_id = basename[basename.find('_')+1:]
            # If it's a known habitat ID, add the file to the habitat data
            if habitat_id in csv_data:
                # Not worried about names between habitat shapefile and csv
                # but about the corresponding ID's. If thats' good, go ahead
                # and make that connection
                csv_data[habitat_id].append(
                    os.path.join(habitats_directory_uri, entry))
            else:
                raise Exception('A habitat layer : %s, with habitat ID: %s '
                    'did not have a corresponding CSV entry in csv file : %s' %
                    (entry, habitat_id, habitats_csv_uri))

    return csv_data

