# -*- coding: utf-8 -*-
# Copyright (c) 2013-2015, Sebastian Linke

# Released under the Simplified BSD license
# (see LICENSE file for details).

"""
Classes and functions providing the core functionality of `shcol`.
"""

from __future__ import unicode_literals
import collections
import itertools

from . import config, helpers

__all__ = ['columnize']

LineProperties = collections.namedtuple(
    'LineProperties', 'column_widths, spacing, num_lines'
)

def columnize(
    items, spacing=config.SPACING, line_width=config.LINE_WIDTH,
    make_unique=config.MAKE_UNIQUE, sort_items=config.SORT_ITEMS,
    decode=config.NEEDS_DECODING
):
    """
    Return a columnized string based on `items`. The result is similar to
    the output generated by tools like `ls`.

    `spacing` should be a non-negative integer defining the number of blank
    characters between two columns.

    `line_width` should be a non-negative integer defining the maximal amount
    of characters that fit in one line. If this is `None` then the terminal's
    width is used.

    If `make_unique` is `True` then only the first occurrence of an item is
    processed and any other occurrences of that item are ignored. This option
    converts whatever is iterated by `items` to a list. You probably don't want
    to use this option when `items` are a mapping.

    If `sort_items` is `True`, then a locale-aware sorted version of `items`
    is used to generate the columnized output. Note that enabling sorting is not
    thread-safe because it temporarily changes the interpreter's global locale
    configuration.

    `decode` defines whether non-Unicode items should be decoded to Unicode. If
     running on Python 3.x then most of the time the "decoding step" is not
     necessary and can be skipped to safe some time. This is why the default
     value for `decode` is `False` when running on a Python 3.x interpreter
     while it is set to `True` when running on Python 2.x.
    """
    if make_unique:
        items = list(helpers.make_unique(items))
    formatter = get_formatter_class(items).for_line_config(spacing, line_width)
    return formatter.format(items, sort_items, decode)

def get_formatter_class(items):
    """
    Return an appropriated class based on the type of `items`.

    If `items` is a dict-like object then `MappingFormatter` will be returned.
    Otherwise, `IterableFormatter` is returned.

    Note that these heuristics are based on rough assumptions. There is no
    guarantee that formatting with the returned class will not fail.
    """
    if helpers.is_mapping(items):
        return MappingFormatter
    return IterableFormatter


class IterableFormatter(object):
    """
    A class to do columnized formatting on a given iterable of strings.
    """
    def __init__(
        self, calculator, linesep=config.LINESEP, encoding=config.ENCODING
    ):
        """
        Initialize the formatter.

        `calculator` will be used to determine the width of each column when
        columnized string formatting is done. It should be a class instance
        that implements a `.get_properties()` method in a similar way as
        `ColumnWidthCalculator` does.

        `linesep` defines the character(s) used to start a new line.

        `encoding` should be a string defining the codec name to be used in
        cases where decoding of items is requested.
        """
        self.calculator = calculator
        self.linesep = linesep
        self.encoding = encoding

    @classmethod
    def for_line_config(
        cls, spacing=config.SPACING, line_width=config.LINE_WIDTH,
    ):
        """
        Return a new instance of this class with a pre-configured calculator.
        The calculator instance will be based on the given `spacing` and
        `line_width` parameters.
        """
        calculator = ColumnWidthCalculator(
            spacing, line_width, allow_exceeding=(line_width is None)
        )
        return cls(calculator)

    def format(
        self, items, sort_items=config.SORT_ITEMS, decode=config.NEEDS_DECODING
    ):
        """
        Return a columnized string based on `items`.

        `sort_items` should be a boolean defining whether `items` should be
        sorted before they are columnized.

        `decode` defines whether each item should be decoded in order to get
        Unicode-strings. When passing items that already are Unicode then this
        parameter may be set to `False` to skip the "decoding step" (which will
        safe some time).

        Please note that Python 2.x (byte-)strings with non-ascii characters in
        it (e.g. German umlauts) should always be decoded. Otherwise, formatting
        is likely to return unexpected results. Unicode items in Python 2.x are
        *not* affected by this.
        """
        if sort_items:
            items = self.get_sorted(items)
        if decode:
            items = self.get_decoded(items)
        lines = self.make_lines(items)
        return self.linesep.join(lines)

    @staticmethod
    def get_sorted(items):
        """
        Return a sorted version of `items`.
        """
        return helpers.get_sorted(items)

    def get_decoded(self, items):
        """
        Return a decoded version of `items`.
        """
        return list(helpers.get_decoded(items, self.encoding))

    def make_lines(self, items):
        """
        Return columnized lines for `items` yielded by an iterator. Note that
        this method does not append extra newline characters to the end of the
        resulting lines.
        """
        props = self.get_line_properties(items)
        line_chunks = self.make_line_chunks(items, props)
        template = self.make_line_template(props)
        for chunk in line_chunks:
            try:
                yield template % chunk
            except TypeError:
                # cached template does not match current chunk length
                # -> regenerate and try again
                template = self.make_line_template(props, len(chunk))
                yield template % chunk

    def get_line_properties(self, items):
        """
        Return a `LineProperties`-instance with a configuration based on given
        `items`.
        """
        return self.calculator.get_line_properties(items)

    @staticmethod
    def make_line_chunks(items, props):
        """
        Return an iterable of tuples that represent the elements of `items` for
        each line meant to be used in a formatted string. Note that the result
        depends on the value of `props.num_lines` where `props` should be a
        `LineProperties`-instance.
        """
        return [
            tuple(items[i::props.num_lines]) for i in range(props.num_lines)
        ]

    def make_line_template(self, props, num_columns=None):
        """
        Return a string meant to be used as a formatting template for *one* line
        of columnized output. The template will be suitable for old-style string
        formatting ('%s' % my_string).

        `props` is expected to be a `LineProperties`-instance as it is defined
        in this module. Appropriated format specifiers are generated based on
        the information of `props.column_widths`. In the resulting template the
        specifiers are joined by using a separator with a `props.spacing` number
        of blank characters.

        `num_columns` defines the number of columns that the resulting template
        should cover. If `None` is used then all items of `props.columns_widths`
        are taken into account. Otherwise, the resulting format string will only
        hold specifiers for the first `num_columns`.
        """
        widths = props.column_widths[:num_columns]
        if not widths:
            return ''
        parts = [self.get_padded_template(width) for width in widths[:-1]]
        parts.append(self.get_unpadded_template(widths[-1]))
        return (props.spacing * ' ').join(parts)

    @staticmethod
    def get_padded_template(width):
        """
        Return a column template suitable for string formatting with exactly
        one string argument (e.g. template % mystring).

        The template is guaranteed to produce results which are always exactly
        `width` characters long. If a string smaller than the given `width` is
        passed to the template then the result will be padded on its right side
        with as much blank characters as necessary to reach the required
        `width`. In contrast, if the string is wider than `width` then all
        characters on the right side of the string which are "too much" are
        truncated.
        """
        return '%%-%d.%ds' % (width, width)

    @staticmethod
    def get_unpadded_template(width):
        """
        Same as `get_padded_template()` but does not pad blank characters if
        the string passed to the template is shorter than the given `width`.
        """
        return '%%.%ds' % width


class MappingFormatter(IterableFormatter):
    """
    A class to do columnized formatting on a given mapping of strings.
    """
    @classmethod
    def for_line_config(
        cls, spacing=config.SPACING, line_width=config.LINE_WIDTH
    ):
        """
        Return a new instance of this class with a pre-configured calculator.
        The calculator instance will be based on the given `spacing` and
        `line_width` parameters.
        """
        calculator = ColumnWidthCalculator(
            spacing, line_width, num_columns=2, allow_exceeding=False
        )
        return cls(calculator)

    @staticmethod
    def get_sorted(mapping):
        """
        Return a sorted version of `mapping`.
        """
        return helpers.get_dict(
            (key, mapping[key]) for key in helpers.get_sorted(mapping.keys())
        )

    def get_decoded(self, mapping):
        """
        Return a decoded version of `mapping`.
        """
        keys = helpers.get_decoded(mapping.keys(), self.encoding)
        values = helpers.get_decoded(mapping.values(), self.encoding)
        return helpers.get_dict(zip(keys, values))

    def get_line_properties(self, mapping):
        """
        Return a `LineProperties`-instance with a configuration based on the
        strings in the keys and values of `mapping`.
        """
        items = itertools.chain(mapping.keys(), mapping.values())
        return self.calculator.get_line_properties(items)

    @staticmethod
    def make_line_chunks(mapping, props):
        """
        Return an iterable of tuples that represent the elements of `mapping`
        for each line meant to be used in a formatted string. Note that the
        result depends on the value of `props.num_lines` where `props` should
        be a `LineProperties`-instance.
        """
        return list(mapping.items())[:props.num_lines]


class ColumnWidthCalculator(object):
    """
    A class with capabilities to calculate the widths for an unknown number
    of columns based on a given sequence of strings.
    """
    def __init__(
        self, spacing=config.SPACING, line_width=config.LINE_WIDTH,
        num_columns=None, allow_exceeding=True
    ):
        """
        Initialize the calculator.
        
        `spacing` defines the number of blanks between two columns.
        
        `line_width` is the maximal amount of characters that fit in one line.
        If this is `None` then the terminal's width is used.

        `num_columns` defines a fixed number of columns to be used for column
        width calculation. This can be `None` to let the calculator decide about
        the number of columns on its own. Note that the "`None`-mode" is often
        useful when the input is just something like a list of names where the
        resulting number of columns fitting in one line is usually unknown,
        while the "fixed-mode" make sense when you have structured input where
        the number of resulting columns is essential (e.g. in mappings).
        
        If `allow_exceeding` is set to `True` then the calculator is allowed to
        exceed the given line width in cases where `num_columns` is `None` *and*
        at least one item of the input is wider than the allowed line width. The
        result will then consist of only one column and that column's width will
        equal to the widest item's width. Note that in all other constellations
        a `ValueError` will be raised instead.
        """
        self.spacing = helpers.num(spacing)
        self._line_width = helpers.num(line_width)
        self.num_columns = helpers.num(num_columns)
        self.allow_exceeding = allow_exceeding

    @property
    def line_width(self):
        """
        Return the line width used by this calculator.
        """
        if self._line_width is None:
            try:
                return helpers.get_terminal_width()
            except (IOError, OSError):
                return config.LINE_WIDTH_FALLBACK
        return self._line_width

    @line_width.setter
    def line_width(self, width):
        """
        Set the line width to be used by this calculator.
        """
        self._line_width = width

    def get_line_properties(self, items):
        """
        Return a namedtuple containing meaningful properties that may be used
        to format `items` as a columnized string.

        The members of the tuple are: `column_widths`, `spacing`, `num_lines`.
        """
        item_widths = [len(item) for item in items]
        column_widths, num_lines = self.calculate_columns(item_widths)
        return LineProperties(column_widths, self.spacing, num_lines)

    def calculate_columns(self, item_widths):
        """
        Calculate column widths based on `item_widths`, expecting `item_widths`
        to be a sequence of non-negative integers that represent the length of
        each corresponding string. The result is returned as a tuple consisting
        of two elements: A sequence of calculated column widths and the number
        of lines needed to display all items when using that information to do
        columnized formatting.

        Note that this instance's `.line_width` and `.spacing` attributes are
        taken into account when calculation is done. However, the column widths
        of the resulting tuple will *not* include that spacing.
        """
        if not item_widths:
            return [], 0
        if self.num_columns is None:
            max_columns = self.calculate_max_columns(item_widths)
            candidates = self.get_column_configs(item_widths, max_columns)
            for column_widths, num_lines in candidates:
                if self.fits_in_line(column_widths):
                    return column_widths, num_lines
            column_widths, num_lines = [max(item_widths)], len(item_widths)
        else:
            column_widths, num_lines = self.get_widths_and_lines(
                item_widths, self.num_columns
            )
        if not self.allow_exceeding and not self.fits_in_line(column_widths):
            raise ValueError('items do not fit in line')
        return column_widths, num_lines

    def calculate_max_columns(self, item_widths):
        """
        Return the number of columns that is guaranteed not to be exceeded when
        `item_widths` are columnized. Return `0` if `item_widths` is empty.

        This is meant to be used as an upper bound on which the real calculation
        of resulting column widths may be based on. Using this value can save a
        remarkable number of iterations depending on the way how column width
        calculation is implemented.
        """
        num_items = len(item_widths)
        if num_items <= 1:
            return num_items
        smallest_item, widest_item = min(item_widths), max(item_widths)
        if widest_item >= self.line_width:
            return 1
        remaining_width = self.line_width - widest_item
        min_width = self.spacing + smallest_item
        possible_columns = 1 + remaining_width // min_width
        return min(num_items, possible_columns)

    def get_column_configs(self, item_widths, max_columns):
        """
        Return an iterator that yields a sequence of "column configurations" for
        `item_widths`. A configuration is a 2-element tuple consisting of a list
        of column widths and the number of lines that are needed to display all
        items with that configuration. The maximum number of columns is defined
        by `max_columns`.
        
        Note that `max_columns` is also used to define the initial amount of
        columns for the first configuration. Subsequent configurations are
        calculated by decreasing that amount at each step until an amount of
        zero columns is reached.

        Depending on the underlying algorithm this method must not necessarily
        return each possible configuration. In fact, the current implementation
        prefers balanced column lengths where only the last column is allowed to
        be shorter than the other columns. This might result in ommiting some
        configurations. See `.get_widths_and_lines()` for details.
        """
        cached_config = ()
        for num_columns in range(max_columns, 0, -1):
            config = self.get_widths_and_lines(item_widths, num_columns)
            if config != cached_config:
                yield config
            cached_config = config

    @staticmethod
    def get_widths_and_lines(item_widths, max_columns):
        """
        Calulate column widths based on `item_widths` for an amount of at most
        `max_columns` per line. The resulting column widths are represented
        as a list of non-negative integers. This list and the number of lines
        needed to display all items is returned as a 2-element tuple.

        Note that this method does *not* check whether the resulting column
        width configuration would fit in a line. Also it does *not* take any
        spacing into account.

        This algorithm prefers balanced column lengths over matching exactly
        `max_columns`. Only the last column is allowed to be shorter than the
        other columns. This means that you might encounter results which have a
        much fewer number of columns than you had requested. In consequence of
        this, two requests with different `max_columns` might return the same
        result.
        """
        num_items = len(item_widths)
        max_columns = helpers.num(max_columns)  # ensure non-negative integer
        num_lines = num_items // max_columns + bool(num_items % max_columns)
        column_widths = [
            max(item_widths[i : i + num_lines])
            for i in range(0, num_items, num_lines)
        ]
        return column_widths, num_lines

    def fits_in_line(self, column_widths):
        """
        Summarize the values of given `column_widths`, add `.spacing` between
        each column and then check whether it exceeds `.line_width`. Return
        `True` if the result does *not* exceed the allowed line width. Return
        `False` otherwise.
        """
        total = sum(column_widths) + (len(column_widths) - 1) * self.spacing
        return total <= self.line_width
