# -*- coding: utf-8 -*-
#
# Copyright (c) 2007 - 2009 -- Lars Heuer - Semagia <http://www.semagia.com/>.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
#     * Redistributions of source code must retain the above copyright
#       notice, this list of conditions and the following disclaimer.
#
#     * Redistributions in binary form must reproduce the above
#       copyright notice, this list of conditions and the following
#       disclaimer in the documentation and/or other materials provided
#       with the distribution.
#
#     * Neither the name of the project nor the names of the contributors 
#       may be used to endorse or promote products derived from this 
#       software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
# LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
"""\
Provides deserialization of TM/XML topic maps.

:author:       Lars Heuer (heuer[at]semagia.com)
:organization: Semagia - http://www.semagia.com/
:version:      $Rev: 169 $ - $Date: 2009-06-26 14:44:17 +0200 (Fr, 26 Jun 2009) $
:license:      BSD license
"""
from xml.sax import handler as sax_handler
from tm import TMDM, XSD
from tm import mio
from tm.mio.xmlutils import attributes
from tm.irilib import resolve_iri

# Disable errors that the AttrsImpl has no 'get' method
# pylint: disable-msg=E1103

__all__ = ['TMXMLContentHandler']

_NS_TMXML = 'http://psi.ontopia.net/xml/tm-xml/'
_NS_TMDM = 'http://psi.topicmaps.org/iso13250/model/'
_TOPIC_NAME = mio.SUBJECT_IDENTIFIER, TMDM.topic_name   # Default name type

# States
_INITIAL = 0
_TOPICMAP = 1
_TOPIC = 2
_IDENTITY = 3
_ASSOCIATION = 4
_AFTER_ROLE = 5
_BASENAME = 6
_NAME = 7
_VARIANT = 8
_PROPERTY = 9

_ATTR_ID = None, 'id'
_ATTR_ROLE = None, 'role'
_ATTR_SCOPE = None, 'scope'
_ATTR_REIFIER = None, 'reifier'
_ATTR_DATATYPE = None, 'datatype'
_ATTR_TOPIC_REF = None, 'topicref'
_ATTR_OTHER_ROLE = None, 'otherrole'


class TMXMLContentHandler(sax_handler.ContentHandler):
    """\
    Content handler for TM/XML topic maps.
    """
    def __init__(self):
        sax_handler.ContentHandler.__init__(self)
        self.map_handler = None
        self._prefixes = {}
        self._content = []
        self._state = _INITIAL
        self.doc_iri = None
        self._current_topic = None
        self._type = None
        self._scope = None
        self._reifier = None
        self._datatype = None
        self._seen_identity = False

    def startDocument(self):
        self._prefixes = {}
        self._content = []
        self._state = _INITIAL
        self._current_topic = None
        self._type = None
        self._scope = None
        self._reifier = None
        self._datatype = None

    def startPrefixMapping(self, prefix, uri):
        self._prefixes[prefix] = uri

    def endPrefixMapping(self, prefix):
        del self._prefixes[prefix]

    def startElementNS(self, (uri, name), qname, attrs):
        attrs = attributes(attrs)
        state = self._state
        handler = self.map_handler
        topic_ref = self._topic_by_reference
        if _INITIAL is state:
            self._handle_reifier(attrs)
            state = _TOPICMAP
        elif _TOPICMAP is state:
            tid = attrs.get(_ATTR_ID, None)
            # The topic id is optional iff the topic has at least one sid / slo
            if tid:
                self._current_topic = self._resolve_iid(tid)
                handler.startTopic(self._current_topic)
                self._seen_identity = True
            else:
                self._seen_identity = False
            type_ = self._resolve_type(uri, name)
            if type_:
                if self._seen_identity:
                    handler.isa(type_)
                else:
                    # No startTopic event, delay the type-instance assoc.
                    self._type = type_
            state = _TOPIC
        elif _TOPIC is state:
            if uri == _NS_TMXML and name in ('identifier', 'locator'):
                state = _IDENTITY
            elif attrs.get(_ATTR_ROLE, None):
                handler.startAssociation(self._resolve_type(uri, name))
                self._handle_scope(attrs)
                self._handle_reifier(attrs)
                # 1 .. n-ary
                handler.startRole(topic_ref(attrs.getValue(_ATTR_ROLE)))
                handler.player(self._current_topic)
                handler.endRole()
                if attrs.get(_ATTR_TOPIC_REF, None):
                    # 2-ary
                    handler.startRole(topic_ref(attrs.getValue(_ATTR_TOPIC_REF)))
                    handler.player(topic_ref(attrs.getValue(_ATTR_OTHER_ROLE)))
                    handler.endRole()
                state = _ASSOCIATION
            else:
                self._scope = self._collect_scope(attrs)
                self._type = self._resolve_type(uri, name)
                self._reifier = attrs.get(_ATTR_REIFIER, None)
                self._datatype = attrs.get(_ATTR_DATATYPE, XSD.string)
                state = _PROPERTY
        elif _ASSOCIATION is state:
            handler.startRole(self._resolve_type(uri, name))
            handler.player(topic_ref(attrs.getValue(_ATTR_TOPIC_REF)))
            self._handle_reifier(attrs)
            handler.endRole()
            state = _AFTER_ROLE
        elif _PROPERTY is state and uri == _NS_TMXML and name == 'value':
            self._content = []
            state = _BASENAME
        elif _NAME is state and uri == _NS_TMXML and name == 'variant':
            self._scope = self._collect_scope(attrs)
            self._reifier = attrs.get(_ATTR_REIFIER, None)
            self._datatype = attrs.get(_ATTR_DATATYPE, XSD.string)
            state = _VARIANT
        self._state = state

    def endElementNS(self, name, qname):
        state = self._state
        handler = self.map_handler
        uri, name = name
        if _TOPICMAP is state:
            state = _INITIAL
        elif _TOPIC is state:
            handler.endTopic()
            state = _TOPICMAP
        elif _IDENTITY is state:
            iri = self._resolve_iri(''.join(self._content))
            if not self._seen_identity:
                self._handle_delayed_topic(name, iri)
            elif name == 'identifier':
                handler.subjectIdentifier(iri)
            else:
                assert name == 'locator'
                handler.subjectLocator(iri)
            self._content = []
            state = _TOPIC
        elif _PROPERTY is state:
            handler.startOccurrence(self._resolve_type(uri, name))
            handler.value(''.join(self._content), self._datatype)
            self._process_scope(self._scope)
            self._process_reifier(self._reifier)
            handler.endOccurrence()
            self._content = []
            state = _TOPIC
        elif _BASENAME is state:
            handler.startName(self._type)
            self._process_scope(self._scope)
            self._process_reifier(self._reifier)
            handler.value(''.join(self._content))
            self._content = []
            state = _NAME
        elif _NAME is state:
            handler.endName()
            state = _TOPIC
        elif _VARIANT is state:
            handler.startVariant()
            self._process_scope(self._scope)
            self._process_reifier(self._reifier)
            handler.value(''.join(self._content), self._datatype)
            handler.endVariant()
            self._content = []
            state = _NAME
        elif _ASSOCIATION is state:
            handler.endAssociation()
            state = _TOPIC
        elif _AFTER_ROLE is state:
            state = _ASSOCIATION
        self._state = state

    def characters(self, content):
        if self._state in (_IDENTITY, _PROPERTY, _BASENAME, _VARIANT):
            self._content.append(content)

    def _handle_delayed_topic(self, name, iri):
        """\
        Sends a startTopic event with a subject identifier or subject locator.
        """
        if name == 'identifier':
            self._current_topic = mio.SUBJECT_IDENTIFIER, iri
        elif name == 'locator':
            self._current_topic = mio.SUBJECT_LOCATOR, iri
        self.map_handler.startTopic(self._current_topic)
        if self._type:
            self.map_handler.isa(self._type)
            self._type = None

    def _topic_by_reference(self, ref):
        """\
        Resolves a topic either by its subject identifier or item identifier.
        If `ref` is a QName (prefix:local), a subject identifier is returned, 
        otherwise an item identifier.
        """
        if ':' in ref:
            return self._resolve_sid(ref)
        return self._resolve_iid(ref)

    def _resolve_iid(self, ref):
        """\
        Resolves the specified `ref` against the base IRI and returns an 
        item identifier reference.
        """
        return mio.ITEM_IDENTIFIER, self._resolve_iri('#%s' % ref)

    def _resolve_sid(self, qname):
        """\
        Resolves the specified `qname` against the base IRI and returns a
        subject identifier reference.
        """
        prefix, local = qname.split(':')
        try:
            return mio.SUBJECT_IDENTIFIER, self._resolve_iri(self._prefixes[prefix] + local)
        except KeyError:
            raise mio.MIOException('Undefined prefix "%s"' % prefix)

    def _resolve_iri(self, iri):
        """\
        Resolves the specified IRI against the base IRI.
        """
        return resolve_iri(self.doc_iri, iri)

    def _resolve_type(self, uri, name):
        """\
        Returns a topic reference (subject identifier / item identifier) from 
        the specified (uri, name) tuple. Returns ``None`` iff the tuple
        is ('http://psi.topicmaps.org/iso13250/model/', 'topic') 
        """
        if not uri:
            return self._resolve_iid(name)
        if uri == _NS_TMDM:
            if name == 'topic-name':
                return _TOPIC_NAME
            elif name == 'topic':
                return None
        return mio.SUBJECT_IDENTIFIER, self._resolve_iri(uri + name)

    def _collect_scope(self, attrs):
        """\
        Returns topic references from the specified attributes.
        
        If the attributes provide no 'scope' key an empty collection is returned.
        """
        scope = attrs.get(_ATTR_SCOPE, None)
        if not scope:
            return ()
        topic_ref = self._topic_by_reference
        return [topic_ref(theme) for theme in scope.split()]

    def _handle_scope(self, attrs):
        """\
        Retrieves the ``scope`` attribute from the attributes and sends 
        ``startScope`` .. ``endScope`` notifications if the ``scope`` attribute
        is given.
        """
        self._process_scope(self._collect_scope(attrs))

    def _process_scope(self, themes):
        """\
        Sends ``startScope`` .. ``endScope`` notifications iff `themes` is not
        empty.
        """
        if themes:
            handler = self.map_handler
            handler.startScope()
            for theme in themes:
                handler.theme(theme)
            handler.endScope()

    def _handle_reifier(self, attrs):
        """\
        Retrieves the ``reifier`` attribute from the attributes and sends
        ``startReifier`` .. ``endReifier`` notifications iff the attribute is
        given.
        """
        self._process_reifier(attrs.get(_ATTR_REIFIER, None))

    def _process_reifier(self, reifier):
        """\
        Resolves the ``reifier`` reference into an absolute IRI (either
        subject identifier or item identifier) iff ``reifier`` is not ``None``
        and notifies the handler about the reifier.
        """
        if reifier:
            self.map_handler.reifier(self._topic_by_reference(reifier))

