# encoding=utf8
import re
import os
import time
import uuid
from datetime import datetime
from .db import escape, Dict
from sqlalchemy.types import (
    Integer,
    DateTime,
    Time,
    String,
    Boolean
)
from openpyxl import Workbook
from .xlsx import create_sheet
from .jsql import JsonSqlParser

import logging
log = logging.getLogger(__name__)


def normalize_sql(sql, ctx):
    params = {}
    matched = re.findall(r'\$\w+\.{0,1}\w+', sql)
    for m in matched:
        bb = m.split('.')
        obj_key, prop = bb[0][1:], None # remove first $

        if len(bb) > 1:
            prop = bb[1]

        param_name = obj_key
        val = ctx.get(obj_key)
        if not val:
            continue

        if prop:
            try:
                param_name = f'{param_name}_{prop}'
                val = val.get(prop)
            except:
                continue

        sql = sql.replace(m, f':{param_name}')
        params[param_name] = val
    return sql, params


def normalize_create_entity(user, data, info):
    res = {}
    props = info.props
    for key in props:
        prop = props[key]
        if prop.get('createUserEnabled'):
            if key not in data:
                data[key] = user['id']
        if prop.get('createOrgEnabled'):
            if key not in data:
                data[key] = user['orgId']
        if prop.get('createTimeEnabled'):
            if key not in data:
                data[key] = datetime.now()
        if prop.get('defaultValue'):
            if key not in data:
                data[key] = prop.get('defaultValue')
        auto_gen = prop.get('autoGen')
        if auto_gen:
            if auto_gen == 'uuid':
                data[key] = str(uuid.uuid4())
            elif auto_gen == 'timestamp':
                data[key] = datetime.now().strftime('%Y%m%d%H%M%S')
            else:
                pass  # add more support

        if prop.get('validateEnabled'):  # only check availability
            if key not in data:
                msg = prop.get('validateFailureMsg') or f'{key} is required'
                res[key] = msg
    return res


def normalize_create_entity_list(user, data, info):
    if not isinstance(data, (tuple, list)):
        data = [data]
    res = []
    for d in data:
        res_i = normalize_create_entity(user, d, info)
        if len(res_i) > 0:
            res.append(res_i)
    return res


def normalize_update_entity(user, data, info):
    props = info.props
    for key in props:
        prop = props[key]
        if prop.get('updateUserEnabled'):
            data[key] = user.id
        if prop.get('updateTimeEnabled'):
            data[key] = datetime.now()


def normalize_update_entity_list(user, data, info):
    if not isinstance(data, (tuple, list)):
        data = [data]
    for d in data:
        normalize_update_entity(user, d, info)


def primary_key_cond(data, t):
    conds = []
    params = {}
    error = {}
    for c in t.primary_key:
        key = c.name
        if key not in data:
            error[key] = f"{key} is required"
            continue

        cond = f"{escape(t.name)}.{escape(key)}=:{key}"
        conds.append(cond)
        params[key] = data[key]

    return conds, params, error


def primary_key_cond_list(data, t):
    """批量更新仅支持默认id为主键"""
    error = {}
    key = 'id'
    param_name = f"{t.name}_ids"
    cond = f"{escape(t.name)}.id in :{param_name}"
    param_list = []
    for d in data:
        if key not in d:
            error[key] = f"{key} is required"
            break
        param_list.append(d['id'])
    return [cond], {param_name: param_list}, error


def filter_arg(args, filter_p, main_table):
    t, c = filter_p
    key = f'{t}.{c}'
    if key in args:
        return args[key]
    if t == main_table and c in args:  # current table default
        return args[c]


class SqlBlock:
    def __init__(self):
        self.select_columns = []
        self.export_columns = []  # 导出SQL字段
        self.join_tables = []
        self.search_columns = []
        self.match_columns = []
        self.filter_columns = []
        self.filter_params = []
        self.orderby_columns = []
        self.groupby_columns = []

        # 模型配置为子查询SQL的时候，对应默认的条件和参数
        self.conds = []
        self.params = {}

        self.metadata = None
        self.export_props = []  # 导出对象属性
        self.distinct_enabled = False

    def sorted_join_tables(self):
        join_tables = sorted(self.join_tables, key=lambda x: x[1])
        return [t[0] for t in join_tables]


class SqlResult:
    def __init__(self):
        self.distinct = ''
        self.columns = []
        self.export_columns = []
        self.tables = []
        self.conds = []
        self.orderby = []
        self.params = {}
        self.groupby = []

        self.metadata = None
        self.export_props = []  # 导出对象属性

    def to_sql(self, export=False):
        columns = ','.join(self.columns)
        if export:
            columns = ','.join(self.export_columns)

        tables = ' '.join(self.tables)
        sql = f'select {self.distinct} {columns} from {tables}'
        if len(self.conds) > 0:
            cond_str = ' and '.join(self.conds)
            sql = f'{sql} where {cond_str}'
        if len(self.groupby) > 0:
            group_str = ' , '.join(self.groupby)
            sql = f'{sql} group by {group_str}'
        if len(self.orderby) > 0:
            distint_orders = []
            for order in self.orderby:
                if order in distint_orders:
                    continue
                distint_orders.append(order)
            orderby = ' , '.join(distint_orders)
            sql = f'{sql} order by {orderby}'
        return sql


class PermCheckResult:
    def __init__(self):
        self.info = None
        self.status = None
        self.acl = None
        self.message = None


class ModelInfo:
    def __init__(self):
        self.model = None
        self.main_table = None
        self.tables = {}
        self.props = {}
        self.join_tables = []

    def main_table_name(self):
        if not self.model:
            return
        return self.model['alias'] or self.model['table']

    def table_name(self, table):
        if table == self.model['table']:
            return self.model['alias'] or table
        for t in self.join_tables:
            if table == t['table']:
                return table
        return table

    def db_table(self, alias):
        if alias == self.model['table'] or alias == self.model['alias']:
            name = self.model['table']
            return self.tables.get(name)
        for t in self.join_tables:
            if alias == t['table'] or alias == t['alias']:
                name = t['table']
                return self.tables.get(name)


class DbPlus:
    def __init__(self, db):
        self.db = db

    def session(self):
        return self.db.session()

    def model_info(self, entity, user=None, session=None):
        info = ModelInfo()
        sql = f"""
        select
            _model.*,
            mt.table as 'mt.table',
            mt.columns as 'mt.columns',
            mt.join as 'mt.join',
            mt.alias as 'mt.alias',
            mt.order as 'mt.order'
        from
            _model
        left join _model_table mt on _model.model = mt.model
        where _model.model=:model
        order by mt.order asc
        """
        model_tables = self.db.query(sql, model=entity, session=session)
        if not model_tables:
            return None

        info.model = model = Dict()
        tables = {}
        for r in model_tables:
            model['id'] = r['id']
            model['model'] = r.get('model')
            model['table'] = r.get('table')
            model['alias'] = r.get('alias')
            model['where'] = r.get('where')
            model['groupBy'] = r.get('groupBy')
            model['orderBy'] = r.get('orderBy')
            model['sql'] = r.get('sql')
            model['spName'] = r.get('spName')
            model['spParams'] = r.get('spParams')
            model['distinctEnabled'] = r.get('distinctEnabled')
            columns = r.get('columns')
            if columns:
                columns = columns.split(',')
            model['columns'] = columns
            model['displayName'] = r['displayName']

            columns = r['mt.columns']
            if columns:
                columns = columns.split(',')
            table = {
                'table': r['mt.table'],
                'columns': columns,
                'alias': r['mt.alias'],
                'join': r['mt.join'],
                'order': r['mt.order']
            }
            if not table['table']:
                continue
            t = self.db.tables.get(table['table'])
            if t is None:
                continue
            tables[t.name] = t
            if table['alias']:
                tables[table['alias']] = t
            info.join_tables.append(table)

        info.model = model
        info.main_table = self.db.tables.get(model['table'])
        if info.main_table is not None:
            tables[info.main_table.name] = info.main_table
            if info.model['alias']:
                tables[info.model['alias']] = info.main_table

        props = self.db.query(f"select * from _prop where model=:model", model=entity, session=session)
        # add column database type if missing type
        excluded_props = set()
        if user:
            excluded_props = self.acl_prop(user.id, entity)
        filtered_props = []
        for c in props:
            if c.name in excluded_props:
                continue
            filtered_props.append(c)

            if not c.table:
                continue
            t = self.db.tables.get(c.table)
            if t is None:
                continue

            if c.transObjectName and c.transTable: #转义的关联的表需要加入
                trans_t, column = self.table_column(c.transTable)
                if trans_t is None:
                    continue
                tables[c.transObjectName] = trans_t

            if c.dataType:
                continue
            col = t.c.get(c.column)
            if col is not None:
                c.dataType = self.guess_column_type(c)

        info.props = {c.name: c for c in filtered_props}
        info.tables = tables
        return info

    def table_column(self, table_column):
        bb = table_column.split('.')
        if len(bb) != 2:
            return None, None
        table, column = bb
        t = self.db.tables.get(table)
        if t is None:
            return None, None
        return t, column

    @staticmethod
    def guess_column_type(c):
        if c is None:
            return 'string'

        type_name = str(c.type).lower()
        if c.type is String:
            return 'string'
        if c.type is Integer or type_name.startswith('tinyint'):
            if c.name.startswith('is') or c.name.endswith('Enabled'):
                return 'bool'
            return 'integer'
        if c.type is DateTime:
            return 'datetime'
        if c.type is Time:
            return 'time'
        if c.type is Boolean:
            return 'bool'
        return type_name

    def acl_data(self, uid, model, session=None):
        cond = ''
        if model:
            cond = 'a.model = :model and '
        sql = f"""
        select distinct (a.cond), a.join, a.model, a.name, a.order from
            _acl_data a,
            _user_role ur,
            _role_acl_data rd
        where
            {cond}
            ur.userId = :uid 
            and ur.roleId = rd.roleId 
            and rd.aclDataId = a.id
        """
        return self.db.query(sql, model=model, uid=uid, session=session)

    def acl_model(self, uid, model, session=None):
        res = self.acl_func(uid, resource_type='model', resource=model, session=session)
        if len(res) == 0:
            return
        ops = res[0]['ops']
        return {op['funcKey']: op['funcValue'] for op in ops}

    @staticmethod
    def _acl_func_cond(resource_type=None, resource=None):
        conds = []
        resource_cond = ''
        if resource_type:
            conds.append('s.type = :resource_type')
        if resource:
            conds.append('s.resource = :resource')
        if len(conds) > 0:
            resource_cond = ' and '.join(conds)
            resource_cond += ' and '
        return resource_cond

    @staticmethod
    def _acl_func_result(acl):
        res = []
        group = {}
        for a in acl:
            f = group.get(a.resourceId)
            if not f:
                f = group[a.resourceId] = {
                    'resource': a.resource,
                    'type': a.type,
                    'resourceId': a.resourceId,
                    'ops': []
                }
                res.append(f)
            op = {
                'funcKey': a.funcKey,
                'funcValue': a.funcValue,
                'funcId': a.funcId
            }
            f['ops'].append(op)
        return res

    def acl_func(self, uid, resource_type=None, resource=None, session=None):
        resource_cond = self._acl_func_cond(resource_type=resource_type, resource=resource)

        sql = f"""
        select distinct s.resource, s.type, a.funcKey, a.funcValue, a.id as funcId, a.resourceId from
            _acl_func a,
            _resource s,
            _user_role ur,
            _role_acl_func rf
        where
            {resource_cond}
            a.resourceId = s.id
            and ur.userId = :uid
            and ur.roleId = rf.roleId 
            and rf.aclFuncId = a.id
        """
        acl = self.db.query(sql, resource_type=resource_type, resource=resource, uid=uid, session=session)
        return self._acl_func_result(acl)

    def acl_prop(self, uid, model, session=None):
        sql = f"""
        SELECT
            _prop.model,
            _prop.name
        FROM
            _prop
        JOIN _role_prop ON _role_prop.propId = _prop.id
        JOIN _user_role ur ON ur.roleId = _role_prop.roleId AND _role_prop.roleId = ur.roleId AND ur.userId = :uid
        WHERE
            _prop.model = :model
        """
        res = self.db.query(sql, uid=uid, model=model, session=session)
        return set([r['name'] for r in res])

    def build_acl_cond_sql(self, ctx, model):
        user = ctx.get('user')
        if not user:
            raise Exception('user object must included in ctx')
        acl = self.acl_data(user['id'], model)
        all_params = {}
        all_conds = []
        join_table = []
        for a in acl:
            cond = a['cond']
            if cond:
                cond, params = normalize_sql(cond, ctx)
                all_params.update(params)
                all_conds.append(cond)

            join = a['join']
            if join:
                join, params = normalize_sql(join, ctx)
                if join not in join_table:
                    join_table.append((join, a['order']))   # order matters
                    all_params.update(params)
        return all_conds, all_params, join_table

    @staticmethod
    def add_cond(conds, new_conds, rel='and'):
        new_cond_str = f' {rel} '.join(new_conds)
        conds.append(f"({new_cond_str})")

    @staticmethod
    def parse_where_cond(args, info):
        json_cond = args.get('where')
        conds, params = [], {}
        if json_cond:
            parser = JsonSqlParser(info.tables, info.main_table_name())
            cond = parser.parse_cond(params, json_cond)
            if cond:
                conds.append(cond)
        return conds, params

    @staticmethod
    def parse_join_cond(args, info):
        json_cond = args.get('joinCond')
        conds, params = {}, {}
        if json_cond:
            parser = JsonSqlParser(info.tables, info.main_table_name())
            for join_table in json_cond:
                cond = parser.parse_cond(params, json_cond[join_table])
                if cond:
                    conds[join_table] = cond
        return conds, params

    @staticmethod
    def parse_orderby(args, info):
        orderby = args.get('order')
        if orderby:
            parser = JsonSqlParser(info.tables, info.main_table_name())
            return parser.parse_orderby(orderby)

    def build_sql_blocks(self, info, filter_props=None, join_cond={}, ctx=None):
        excluded_props = set()
        if ctx and 'user' in ctx:
            user = ctx['user']
            excluded_props = self.acl_prop(user.id, info.model['model'])
        block = SqlBlock()
        block.metadata = info
        main_table = info.main_table_name()
        main_alias = info.model['alias'] or main_table
        join_table = f'{escape(main_table)} {escape(main_alias)}'
        whole_sql = info.model.get('sql')
        if whole_sql:  # 配置了完整SQL的场景使用子查询 select * from (yyyy) as xxx
            whole_sql, params = normalize_sql(whole_sql, ctx)
            block.params = params
            join_table = f"({whole_sql}) as {main_table}"
        block.join_tables = [(join_table, -1)] # 第一个条件
        for table_info in info.join_tables:
            table = table_info['table']
            word = 'and'
            join = table_info.get('join')
            if not join:
                join = f'left join {escape(table)}'
                word = 'on'
            alias = table_info['alias'] or table
            cond = join_cond.get(alias)
            if cond:
                join = f"{join} {word} {cond}"
            block.join_tables.append((join, table_info['order']))

        default_export_props = []
        props = info.props
        for p in props:
            prop = props[p]
            if filter_props and p not in filter_props:
                continue
            if not prop.table:
                continue
            if prop.name in excluded_props:
                continue

            prop_t = info.tables.get(prop.table)
            if prop_t is None:
                continue
            if not prop.asEnabled and prop.column not in prop_t.c:
                continue

            if '.' in prop.name:
                full_c = prop.name
                select_c = f"{prop.name} as '{prop.name}'"
            else:
                table_name = info.table_name(prop.table)  # 有别名需要翻译
                select_c = full_c = f'{table_name}.{prop.column}'
                if prop.asEnabled:
                    select_c = f"{prop.column} as '{prop.name}'"
                    if prop.column not in prop_t.c:
                        full_c = None

            if prop.get('selectEnabled'):
                if select_c not in block.select_columns:  # may be duplicated
                    block.select_columns.append(select_c)
                    default_export_props.append(prop)
            search_c, export_c, export_prop, match_c, filter_c, filter_p = None, None, None, None, None, None
            if prop.get('searchEnabled'):
                search_c = full_c
            if prop.get('exportEnabled'):
                export_prop = prop
                export_c = select_c
            if prop.get('matchEnabled'):
                match_c = full_c
            if prop.get('filterEnabled'):
                filter_c = full_c
                table_name = info.table_name(prop.table)
                filter_p = (table_name, prop.column)

            if prop.get('orderEnabled'):
                if prop.get('orderAsDefault'):
                    order_dir = prop.get('orderDir')
                    order_dir = 'desc' if order_dir else 'asc'
                    block.orderby_columns.append(f'{full_c} {order_dir}')

            table_column = prop.get('transTable')  # table.column 格式，找到翻译对应的表和列
            to_columns = prop.get('transColumns')
            if table_column and to_columns:  # 链接外键表
                bb = table_column.split('.')
                if len(bb) != 2:
                    continue

                table, column = bb
                trans_t = self.db.tables.get(table)
                if trans_t is None:
                    continue

                tran_object_name = prop.get('transObjectName')
                if not tran_object_name:
                    tran_object_name = table

                to_columns = to_columns.split(',')
                default_search_c = to_columns[0]  # 第一个关联的列默认为搜索匹配的列

                table = escape(table)
                column = escape(column)
                join = f"left join {table} {escape(tran_object_name)} on {main_table}.{p}={tran_object_name}.{column}"
                order = block.join_tables[len(block.join_tables)-1][1] + 1  # 关联表的顺序+1
                block.join_tables.append((join, order))
                c_name = f'{tran_object_name}.{default_search_c}'
                if prop.get('searchEnabled'):
                    search_c = c_name
                if prop.get('matchEnabled'):
                    match_c = c_name
                if prop.get('filterEnabled') or prop.get('transDict'):
                    filter_c = c_name
                    filter_p = (tran_object_name, default_search_c)

                for to_c in to_columns:
                    if to_c not in trans_t.c:
                        continue
                    name = f'{tran_object_name}.{to_c}'
                    c_name = f"{name} as '{name}'"
                    if c_name not in block.select_columns: # may be duplicated
                        block.select_columns.append(c_name)
                    if prop.get('exportEnabled'):
                        if c_name not in block.export_columns:
                            block.export_columns.append(c_name)

            if search_c:
                block.search_columns.append(search_c)
            if export_c:
                block.export_columns.append(export_c)
            if export_prop:
                block.export_props.append(export_prop)
            if match_c:
                block.match_columns.append(match_c)
            if filter_c:
                block.filter_columns.append(filter_c)
                block.filter_params.append(filter_p)

        if len(block.export_props) == 0:  # 未配置导出信息，默认为查询
            block.export_columns = [c for c in block.select_columns]
            block.export_props = [p for p in default_export_props if p.name != 'id']
        return block

    def build_sql(self, user, model, args={}, extra_conds=[], extra_params={}):
        ctx = {
            'user': user
        }
        info = self.model_info(model, user)
        if info is None:
            return {'message': f'业务模型（{model}）不存在'}, 404
        main_table = info.main_table_name()
        filter_props = args.get('columns')  # 由客户端决定
        if filter_props and isinstance(filter_props, (tuple, list)) and len(filter_props) > 0:
            filter_props = set(filter_props)
        else:
            filter_props = None

        join_cond, join_params = self.parse_join_cond(args, info)
        block = self.build_sql_blocks(info, filter_props=filter_props, join_cond=join_cond, ctx=ctx)
        acl_conds, acl_params, acl_join = self.build_acl_cond_sql(ctx, model)

        block.join_tables.extend(acl_join)

        params = {}
        params.update(block.params)
        params.update(join_params)
        params.update(acl_params)

        search_conds = []
        filter_conds = []

        key = args.get('key')
        if key:
            if len(block.search_columns) > 0:
                conds = [f'{s} like :key' for s in block.search_columns]
                search_conds.extend(conds)
                params['key'] = f'%{key}%'

            if len(block.match_columns) > 0:
                conds = [f'{s} = :key_raw' for s in block.match_columns]
                search_conds.extend(conds)
                params['key_raw'] = key

        filter_args = args.get('filters')
        if filter_args and len(filter_args) > 0:
            if len(block.filter_columns) > 0:
                for i in range(len(block.filter_columns)):
                    filter_c = block.filter_columns[i]
                    filter_p = block.filter_params[i]
                    filter_data = filter_arg(filter_args, filter_p, main_table)
                    if filter_data is None:
                        continue
                    if not isinstance(filter_data, (list, tuple)):
                        filter_data = [filter_data]
                    if len(filter_data) == 0:
                        continue
                    filter_key = '_'.join(filter_p)
                    filter_conds.append(f'{filter_c} in :{filter_key}')
                    params[filter_key] = filter_data

        json_conds, json_params = self.parse_where_cond(args, info)

        conds = block.conds
        if len(acl_conds) > 0:
            self.add_cond(conds, acl_conds, 'and')

        if len(search_conds) > 0:
            search_cond_str = ' or '.join(search_conds)
            conds.append(f"({search_cond_str})")

        if len(filter_conds) > 0:
            self.add_cond(conds, filter_conds, 'and')

        if len(json_conds) > 0:
            self.add_cond(conds, json_conds, 'and')
            params.update(json_params)

        if len(extra_conds) > 0:
            self.add_cond(conds, extra_conds, 'and')
            params.update(extra_params)

        orderby = self.parse_orderby(args, info)
        if orderby and len(orderby) > 0:
            block.orderby_columns = orderby + block.orderby_columns  # 外部指定的排序优先

        # 处理model的直接配置
        model = info.model
        model_where = model.get('where')
        model_groupby = model.get('groupBy')
        model_orderby = model.get('orderBy')
        model_distinct = model.get('distinctEnabled')
        if model_where:
            conds.append(f"({model_where})")
        if model_groupby:
            block.groupby_columns.append(model_groupby)
        if model_orderby:
            block.orderby_columns.append(model_orderby)
        if model_distinct:
            block.distinct_enabled = True

        result = SqlResult()
        result.distinct = 'distinct' if block.distinct_enabled else ''
        result.columns = block.select_columns
        result.export_columns = block.export_columns
        result.tables = block.sorted_join_tables()
        result.groupby = block.groupby_columns
        result.orderby = block.orderby_columns
        result.conds = conds
        result.params = params

        result.metadata = block.metadata
        result.export_props = block.export_props

        return result, 200

    def search(self, user, entity, args={}, extra_conds=[], extra_params={}, session=None):
        sql_res, status = self.build_sql(user, entity, args=args, extra_conds=extra_conds, extra_params=extra_params)
        if status != 200:
            return sql_res, status
        sql = sql_res.to_sql()
        res = self.db.query_page(sql, page=args.get('page') or 1,
                                 limit=args.get('limit') or 10,
                                 do_count=args.get('doCount'),
                                 **sql_res.params, session=session)
        return res, 200

    def export(self, user, entity, args={}, extra_conds=[], extra_params={}, session=None):
        sql_obj, status = self.build_sql(user, entity, args=args, extra_conds=extra_conds, extra_params=extra_params)
        if status != 200:
            return sql_obj, status
        if len(sql_obj.export_columns) == 0:
            return {'message': 'No column to export'}, 600

        sql = sql_obj.to_sql(export=True)

        max_total = 10000 # make it configurable
        page_size = 500
        page = args.get('page') or 1
        data = []
        while True:
            res = self.db.query_page(sql, page=page, limit=page_size, do_count=0, **sql_obj.params, session=session)
            if len(res.data) == 0:
                break
            data.extend(res.data)
            page += 1
            if len(data) > max_total:
                break

        info = sql_obj.metadata
        # 读取字典信息翻译
        dict_domains = [info.props[c].get('transDict') for c in info.props if info.props[c].get('transDict')]
        domains = {}
        if len(dict_domains):
            domains = self.find_dict_domains(dict_domains)

        model = info.model['displayName']
        props = sql_obj.export_props
        props.sort(key=lambda p: p.displayOrder)
        headers = [p.displayName for p in props]
        props_names = []
        props_name_dict = {}
        for p in props:
            name = p['name']
            if p.transObjectName and p.transColumns: # 关联表字段，属性名字组合
                show_column = p.transColumns.split(',')[0]
                name = f"{p.transObjectName}.{show_column}"
            props_names.append(name)
            if p.transDict:
                dict_ = domains.get(p.transDict)
                if dict_:
                    props_name_dict[name] = dict_

        for row in data:  # 翻译带字典的
            for col in row:
                dict_ = props_name_dict.get(col)
                if dict_:
                    val = row[col]
                    val_ = dict_.get(f'{val}')
                    if val_:
                        row[col] = val_['label']
        wb = Workbook()
        ws = wb.worksheets[0]
        create_sheet(ws, model, model, headers, props_names, data)

        path = "/tmp/export"
        time_str = time.strftime('%Y%m%d%H%M%S', time.localtime(time.time()))
        name = f"{info.model['model']}_{time_str}.xlsx"
        if not os.path.exists(path):  # 如果路径不存在
            os.makedirs(path)
        file_name = f"{path}/{name}"
        wb.save(file_name)
        wb.close()

        return Dict({'path': path, 'filename': name}), 200

    def query_one(self, user, entity, id, session=None):
        ctx = {
            'user': user
        }

        info = self.model_info(entity, user)
        if info is None:
            return {'message': f'{entity}不存在'}, 404

        t = info.main_table
        main_table = t.name
        block = self.build_sql_blocks(info, ctx=ctx)
        acl_conds, acl_params, acl_join = self.build_acl_cond_sql(ctx, main_table)
        block.join_tables.extend(acl_join)

        if len(block.select_columns) == 0:
            return {'message': f'{entity}无任何模型属性列可查询'}, 404

        columns = ','.join(block.select_columns)
        tables = ' '.join(block.sorted_join_tables())
        sql = f'select {columns} from {tables} where {escape(main_table)}.id=:id'
        if len(acl_conds) > 0:
            acl = ' or '.join(acl_conds)
            sql = f'{sql} and ({acl})'

        return self.db.query_one(sql, id=id, **acl_params, session=session), 200

    def check_perm(self, user, model, op_key, op_name, session=None):
        res = PermCheckResult()
        info = self.model_info(model, user)
        if info is None:
            res.message = {'message': f'{model}不存在'}
            res.status = 404
            return res
        res.info = info
        model_name = info.model['displayName']
        acl = self.acl_model(user.id, model, session=session)
        if not acl or not acl.get(op_key):
            res.message = {'message': f'无功能权限{op_name}{model_name}表单'}
            res.status = 403
            return res
        res.acl = acl
        res.status = 200
        return res

    def create(self, user, model, data, session=None):
        if len(data) == 0:
            return {'message': f'不能创建空表单'}, 400

        check = self.check_perm(user, model, 'createEnabled', '创建')
        if check.status != 200:
            return check.message, check.status
        info = check.info
        t = info.main_table
        main_table = t.name
        res = normalize_create_entity(user, data, info)
        if len(res) > 0:
            return res, 400

        return self.db.add(main_table, data, session=session), 200

    def create_many(self, user, model, data, session=None):
        """
        批量增加，验证创建功能权限
        :param user: 登录用户
        :param model: 业务模型
        :param data: 数据列表
        :return:
        """

        check = self.check_perm(user, model, 'createEnabled', '创建')
        if check.status != 200:
            return check.message, check.status

        if not isinstance(data, (tuple, list)):
            data = [data]

        info = check.info
        t = info.main_table
        main_table = t.name

        res = normalize_create_entity_list(user, data, info)
        if len(res) > 0:
            return res, 400

        t = info.main_table
        return self.db.add_many(main_table, data, session=session), 200

    def update(self, user, model, data, session=None):
        if session:
            return self._update(user, model, data, session=session)
        with self.db.session() as s:
            return self._update(user, model, data, session=s)

    def _update(self, user, model, data, session):
        check = self.check_perm(user, model, 'updateEnabled', '更新', session=session)
        if check.status != 200:
            return check.message, check.status

        info = check.info
        model_name = info.model['displayName']

        normalize_update_entity(user, data, info)

        t = info.main_table
        p_conds, p_params, error = primary_key_cond(data, t)
        if len(error) > 0:
            return error, 400

        ctx = {
            'user': user
        }
        acl_conds, acl_params, acl_join = self.build_acl_cond_sql(ctx, t.name)

        block = self.build_sql_blocks(info, ctx=ctx)
        if len(acl_conds) > 0:
            block.join_tables.extend(acl_join)
            tables = ' '.join(block.sorted_join_tables())
            p_cond = ' and '.join(p_conds)
            acl_cond = ' or '.join(acl_conds)
            sql = f"select {escape(t.name)}.* from {tables} where ({p_cond}) and ({acl_cond})"
            db_entity = session.query_one(sql, **p_params, **acl_params)
            if not db_entity:
                return {'message': f'无数据权限更新{model_name}表单, {p_params}'}, 403
        else:
            tables = ' '.join(block.sorted_join_tables())
            sql = f"select {escape(t.name)}.* from {tables} where {escape(t.name)}.id=:id"
            db_entity = session.query_one(sql, id=data.id)
            if not db_entity:
                return {'message': f'{model_name}(id={data.id})不存在'}, 404
        session.merge(t.name, data)
        db_entity.update(data)
        return db_entity, 200

    def update_many(self, user, model, data, session=None):
        """
        批量更新，验证更新功能权限，批量数据各个对应的数据权限
        :param user: 登录用户
        :param model: 业务模型
        :param data: 数据列表
        :return:
        """
        check = self.check_perm(user, model, 'updateEnabled', '更新')
        if check.status != 200:
            return check.message, check.status

        info = check.info

        if not isinstance(data, (tuple, list)):
            data = [data]
        normalize_update_entity_list(user, data, info)

        t = info.main_table
        p_conds, p_params, error = primary_key_cond_list(data, t)
        if len(error) > 0:
            return error, 400

        ctx = {
            'user': user
        }
        acl_conds, acl_params, acl_join = self.build_acl_cond_sql(ctx, t.name)

        block = self.build_sql_blocks(info, ctx=ctx)
        if len(acl_conds) > 0:
            block.join_tables.extend(acl_join)
            tables = ' '.join(block.sorted_join_tables())
            p_cond = ' and '.join(p_conds)
            acl_cond = ' or '.join(acl_conds)
            sql = f"select {escape(t.name)}.* from {tables} where ({p_cond}) and ({acl_cond})"
            db_entity_list = self.db.query(sql, **p_params, **acl_params)

        else:
            tables = ' '.join(block.sorted_join_tables())
            p_cond = ' and '.join(p_conds)
            sql = f"select {escape(t.name)}.* from {tables} where {p_cond}"
            db_entity_list = self.db.query(sql, **p_params)

        # TODO 检查所有的记录是否都有权限更新
        self.db.update_many(t.name, data)
        return {'message': '批量更新成功！'}, 200

    def delete(self, user, model, id, session=None):
        if session:
            return self._delete(user, model, id, session=session)
        with self.db.session() as s:
            return self._delete(user, model, id, session=s)

    def _delete(self, user, model, id, session):
        check = self.check_perm(user, model, 'deleteEnabled', '删除', session=session)
        if check.status != 200:
            return check.message, check.status

        info = check.info
        model_name = info.model['displayName']

        t = info.main_table
        ctx = {
            'user': user
        }
        acl_conds, acl_params, acl_join = self.build_acl_cond_sql(ctx, t.name)

        block = self.build_sql_blocks(info, ctx=ctx)
        if len(acl_conds) > 0:
            block.join_tables.extend(acl_join)
            acl_cond = ' or '.join(acl_conds)
            tables = ' '.join(block.sorted_join_tables())
            sql = f"select {escape(t.name)}.* from {tables} where {escape(t.name)}.id=:id and ({acl_cond})"
            db_entity = session.query_one(sql, id=id, **acl_params)
            if not db_entity:
                return {'message': f'无数据权限删除{model_name}(id={id})'}, 403
        else:
            tables = ' '.join(block.sorted_join_tables())
            sql = f"select {escape(t.name)}.* from {tables} where {escape(t.name)}.id=:id"
            db_entity = session.query_one(sql, id=id)
            if not db_entity:
                return {'message': f'{model_name}(id={id})不存在'}, 404
        session.delete(t.name, id)

        return db_entity, 200

    def relation(self, user, link_model, from_key=None, from_data=None, to_key=None, to_data=None, session=None):
        cond = [f"{escape(from_key)} in :from_keys", f"{escape(to_key)} in :to_keys"]
        params = {'from_keys': from_data, 'to_keys': to_data}
        res, status = self.search(user, link_model, {}, extra_conds=cond, extra_params=params, session=session)
        if status != 200:
            return res, status
        return res.data, status

    def relation_update(self, user, relation_model, add_data=None, del_data=None, session=None):
        info = self.model_info(relation_model, user)
        if info is None:
            return {'message': f'业务模型（{relation_model}）不存在'}, 404
        t = info.main_table
        if t is None:
            return {'message': f'关联表{t.name}不存在'}, 404

        model_name = info.model['displayName']
        acl = self.acl_model(user.id, relation_model)
        if add_data:
            if not acl or not acl.get('createEnabled'):
                return {'message': f'无功能权增加{model_name}'}, 403
        if del_data:
            if not acl or not acl.get('deleteEnabled'):
                return {'message': f'无功能权限删除{model_name}'}, 403

        with self.db.session() as s:
            if add_data:
                self._update_rel(s, user, add_data, t, info)
            if del_data:
                self._del_rel(s, del_data, t)
        return {'message': 'success'}, 200

    @staticmethod
    def _update_rel(sess, user, add_data, t, info):
        for d in add_data:
            normalize_update_entity(user, d, info)
            valid = True
            conds = []
            for key in d:
                if key not in t.c:
                    valid = False
                    break
                conds.append(f"{escape(key)}=:{key}")
            if not valid:
                continue
            if len(conds) == 0:
                continue
            cond_str = ' and '.join(conds)
            sql = f"select * from {escape(t.name)} where {cond_str}"
            r = sess.query_one(sql, **d)
            if r:  # exists
                continue
            sess.add(t.name, d)

    @staticmethod
    def _del_rel(sess, del_data, t):
        for d in del_data:
            valid = True
            conds = []
            for key in d:
                if key not in t.c:
                    valid = False
                    break
                conds.append(f"{escape(key)}=:{key}")
            if not valid:
                continue
            if len(conds) == 0:
                continue
            cond_str = ' and '.join(conds)
            sql = f"delete from {escape(t.name)} where {cond_str}"
            sess.execute(sql, **d)

    @staticmethod
    def beautify_display_name(name):
        bb = name.split('_')
        cap_bb = []
        for b in bb:
            if len(b) > 0:
                b = str(b[0]).upper() + b[1:]
            cap_bb.append(b)
        return ''.join(cap_bb)

    @staticmethod
    def generate_table_columns(s, model, t, columns=None, alias=None, is_main=True):
        # clean props with table and column invalid
        props = s.query(f"select * from _prop where model=:model", model=model)
        for p in props:
            if t.name != p.table:
                continue
            if p.table and p.column:
                if p.column not in t.c and not p.asEnabled:  # as 可能是统计之类的字段
                    s.execute(f"delete from _prop where id=:id", id=p.id)

        if columns:
            for column in columns:
                as_enabled = 0
                res = re.search(r'[\s]+as[\s]+', column, re.IGNORECASE) # test or exists
                c = None
                if res:
                    name = column[res.span()[1]:].strip()
                    column = column[0:res.span()[0]].strip()
                    as_enabled = 1
                else:
                    name = column
                    if column not in t.c:
                        continue
                    if alias and not is_main:   # 非主表需要加上前缀
                        name = f"{alias}.{name}"
                    c = t.c[column]

                sql = f"select * from _prop where model=:model and {escape('name')}=:name"
                if s.query_one(sql, model=model, name=name):
                    continue
                prop = {
                    'model': model,
                    'name': name,
                    'table': t.name,
                    'column': column,
                    'asEnabled': as_enabled,
                    'dataType': DbPlus.guess_column_type(c),
                    'displayName': DbPlus.beautify_display_name(name)
                }
                s.add('_prop', prop)

        if not is_main:
            return

        for c in t.c:
            name = c.name
            if alias:
                name = f"{alias}.{c.name}"

            if columns and c.name not in columns:
                sql = f"delete from _prop where model=:model and {escape('name')}=:name"
                s.execute(sql, model=model, name=name)
                continue

            sql = f"select * from _prop where model=:model and {escape('name')}=:name"
            if s.query_one(sql, model=model, name=name):
                continue

            prop = {
                'model': model,
                'name': name,
                'table': t.name,
                'column': c.name,
                'dataType': DbPlus.guess_column_type(c),
                'displayName': DbPlus.beautify_display_name(name)
            }
            s.add('_prop', prop)

    def generate_columns(self, model, info):
        with self.db.session() as s:
            self.generate_table_columns(s, model, info.main_table,
                                        columns=info.model['columns'],
                                        alias=None,
                                        is_main=True)  # 主表属性生成需要前缀
            for table in info.join_tables:
                t = info.tables.get(table['table'])
                if t is None:
                    continue
                alias = table['alias'] or t.name  # 辅表属性生成需要前缀
                columns = table['columns']
                self.generate_table_columns(s, model, t, columns=columns, alias=alias, is_main=False)

    def sync_model(self, model):
        try:
            self.db.reflect()
        except:
            pass
        info = self.model_info(model)
        if not info:
            return {'message': f'Model({model}) Not Found'}, 404
        self.generate_columns(model, info)
        return {'message': 'success'}, 200

    def find_dict_domains(self, domains=[]):
        sql = 'select * from _dict'
        params = {}
        if len(domains) > 0:
            sql = f"{sql} where domain in :domains order by {escape('order')}"
            params['domains'] = domains
        data = self.db.query(sql, **params)
        res = {}
        for r in data:
            d = res.get(r['domain'])
            if not d:
                res[r['domain']] = d = {}
            d[r['key']] = {'label': r['value'], 'order': r['order']}
        return res
