#
# this file is part of the software tool BOOT
# URL: freerangefactory.org
# (C) 2012 Fabrizio Tappero
#
import os


def make_xilinx(syn_out_dir, tld_file, vhdl_files, constraints_file,
                       dev_family,dev_device, dev_package, dev_speed):
    ''' make_xilinx(syn_out_dir, tld_file, vhdl_files, constraints_file,
                       dev_family,dev_device, dev_package, dev_speed):
        Generate and save a Xilinx ISE xtclsh script.
    '''

    # formal vhdl file list
    vhdl_files = '[ list ../' + ' ../'.join(vhdl_files) + ' ]'

    # format constraints file list and take ONLY the first one
    if len(constraints_file) == 0:
        constraints_file = ''
    elif len(constraints_file)>0:
        constraints_file = constraints_file[0]
        constraints_file = os.path.basename(constraints_file) # strip whole path
        constraints_file = '../' + constraints_file
    content = '''#
# xil_syn_script.tcl
#
# script to synthesize your design using xtclsh from Xiling ISE
# usage: xtclsh src/build/xil_syn_script.tcl
#
# this file is automatically generated by "boot"
#
# to use this script you need Xilinx ISE 12.x or later
#
# for some help:
#      www.xilinx.com/itp/xilinx10/books/docs/dev/dev.pdf

# output folder
set compile_directory   %s

# top-level desing file
set tld_file            %s

# input source files:
set vhdl_files          %s

# constraint file
set constraints_file    %s

# Xilinx CableServer host PC:
set cableserver_host {}

set proj $tld_file

puts "Running ISE xtclsh script: \\"xil_syn_script.tcl\\" automatically generated"

if { $cableserver_host == "" } {
  puts "Running with the board connected to the local machine.\\n"
} else {
  puts "Running with the board connected to $cableserver_host.\\n"
}

# Set compile directory
if {![file isdirectory $compile_directory]} {
  file mkdir $compile_directory
   }
cd $compile_directory


# Create a new project or open project
set proj_exists [file exists $proj.xise]

if {$proj_exists == 0} {
    puts "Creating a new Xilinx ISE project ..."
    project new $proj.xise

    # Project settings
    project set family  %s
    project set device  %s
    project set package %s
    project set speed   %s

    # Add source files to the project
    foreach filename $vhdl_files {
      xfile add $filename
    }
    xfile add $constraints_file

    # Make sure $source_directory is properly set
    if { ! [catch {set source_directory $source_directory}] } {
      project set "Macro Search Path" $source_directory -process Translate
    }

} else {

    puts "Opening existing Xilinx ISE project"

    # Open the existing project
    project open $proj.xise
}

# Implementation properties options

# TRANSLATE (often good to set this)
project set "Allow Unmatched LOC Constraints" true -process "Translate"
project set "Allow Unmatched Timing Group Constraints" "true" -process "Translate"

# GENERATE PROGRAMMING FILE (good to set this for Xula-200 board)
project set "Unused IOB Pins" "Float" -process "Generate Programming File"
project set "FPGA Start-Up Clock" "JTAG Clock" -process "Generate Programming File"

# MAP
#project set "Map Effort Level" Medium -process map
#project set "Perform Timing-Driven Packing and Placement" true -process map
#project set "Register Duplication" true -process map
#project set "Retiming" true -process map

# PAR
#project set "Place & Route Effort Level (Overall)" Standard
#project set "Extra Effort (Highest PAR level only)" Normal

# Implement Design
puts "Implement Design..."
# process run "Implement Design"
process run "Generate Programming File"
project close

# All done
puts "End of ISE Tcl script.\\n"

# Download .bit file into your FPGA/CPLD device using impact

# impact cannot be directly run via xtclsh, instead
# an impact script file will be created  and run
set impact_script_filename impact_script.scr
set bit_filename $tld_file.bit

if [catch {set f_id [open $impact_script_filename w]} msg] {
  puts "Can't create $impact_script_filename"
  puts $msg
  exit
}

# For Spartan3E starter kit
if { $cableserver_host == "" } {
        # Assume using locally connected board
        puts $f_id "setMode -bscan"
        puts $f_id "setCable -p usb21"

} else {
        # Assume using cableserver on cableserver_host
        puts $f_id "setMode -bscan"
        puts $f_id "setCable -p usb21 -server $cableserver_host"
}
puts $f_id "addDevice -position 1 -file $bit_filename"
puts $f_id "addDevice -p 2 -part xcf04s"
puts $f_id "addDevice -p 3 -part xc2c64a"
puts $f_id "readIdcode -p 1"
puts $f_id "program -p 1"
puts $f_id "quit"
close $f_id

#puts "\\n Switch on the Spartan3E board, connect the USB cable."
#puts -nonewline "  Press Enter when you are ready to download...\\a"
#flush stdout
#
# The "gets" command fails with the following message, if running within
# the ISE Project Navigator GUI.
#
#   channel "stdin" wasn't opened for reading
#
#if [catch {gets stdin ignore_me} msg] {
#  puts "\\n\\n *** $msg"
#  puts " *** Carrying on regardless ...\\n"
#  flush stdout
#}
# run impact script redirecting stdout
# set impact_p [open "|impact -batch $impact_script_filename" r]
# while {![eof $impact_p]} { gets $impact_p line ; puts $line }
# close $impact

# END
''' %('src/build', tld_file, vhdl_files, constraints_file, \
      dev_family, dev_device, dev_package, dev_speed)
    # NOTE: above we have used "build" directory and not syn_out_dir

    try:
        if os.path.isdir(syn_out_dir):
            _fl = os.path.join(syn_out_dir,'xil_syn_script.tcl')
            open(_fl,'w').write(content)
            print 'Successfully generated Xilinx script file:', _fl
        else:
            print 'Xilinx script not generated.'
            return 1
    except:
        print 'Problems in writing, you might have permission problems.'
        return 1

    return 0


