import M2Crypto
from M2Crypto import RSA, X509, ASN1
#from M2Crypto import Rand
from M2Crypto import EVP
import base64
import os
import datetime

now = datetime.datetime.now

from pki.utils import get_secure_filename
from pki.utils import datetime_to_timestamp
from pki.utils import timestamp_to_datetime
from pki.utils import plus_twentyyears

PADDING_TYPE = M2Crypto.RSA.pkcs1_padding

def create_cert_request(pubkey, privkey, digest="md5", **name):
    """Create a certificate request.

    @param pubkey: the EVP.PKey public key to associate with the request
    @type pubkey: EVP.PKey instance

    @param privkey: the EVP.PKey public key to associate with the request
    @type privkey: EVP.PKey instance

    @param digest: Digest method to use for signing, default is md5
    @type digest: string

    @param: **name - The name of the subject of the request, possible
    arguments are:
    C     - Country name
    ST    - State or province name
    L     - Locality name
    O     - Organization name
    OU    - Organizational unit name
    CN    - Common name
    emailAddress - E-mail address

    @returns: The certificate request in an X509.X509Req object
    """
    req = X509.Request()
    subj = req.get_subject()

    for (key,value) in name.items():
        setattr(subj, key, value)

    req.set_pubkey(pubkey)
    req.sign(privkey, digest)
    return req

def create_certificate(req, (issuerCert, issuerKey), serial,
        (notbefore, notafter), digest="md5"):
    """
    Generate a certificate given a certificate request.

    Arguments: req        - Certificate reqeust to use
               issuerCert - The certificate of the issuer
               issuerKey  - The private key of the issuer
               serial     - Serial number for the certificate
               notbefore  - a datetime.datetime instance when the certificate
                            starts being valid
               notafter   - a datetime.datetime instance when the certificate
                            stops being valid
               digest     - Digest method to use for signing, default is md5
    Returns:   The signed certificate in an X509 object

    WARNING: the datetime instances passed for notbefore and notafter should
    be expressed in UTC.
    """

    not_before = ASN1.ASN1_UTCTIME()
    not_before.set_time(datetime_to_timestamp(notbefore))

    not_after = ASN1.ASN1_UTCTIME()
    not_after.set_time(datetime_to_timestamp(notafter))

    cert = X509.X509()
    cert.set_serial_number(serial)
    
    cert.set_not_before(not_before)
    cert.set_not_after(not_after)
    
    cert.set_issuer(issuerCert.get_subject())

    cert.set_subject(req.get_subject())
    cert.set_pubkey(req.get_pubkey())

    cert.sign(issuerKey, digest)
    
    return cert

def as_evp_pkey(rsa_key):
    """Transform an rsa key into an evp_pkey instance
    """
    evp_pkey = EVP.PKey()
    success = evp_pkey.assign_rsa(rsa_key, capture=0)
    # TODO: Test the success
    return evp_pkey

def create_rsa_keyring(passphrase_callback, key_size=2048, exponent=65537):
    """ Return a new RSA keyring as tuple (<public key>, <private key>)
    @param passphrase_callback: A Python callable object that is invoked
    during key generation; its usual purpose is to provide visual
    feedback.
    You must load a randpool file before calling this method with
    Rand.load_file('randpool', -1).
    After calling the method, you must save the randpool file with
    Rand.save_file('randpool')
    @type passphrase_callback: Callback

    @param key_size: Key length, in bits
    @type key_size: Integer

    @param exponent: The RSA public exponent
    @type exponent: Integer
    """
    keyring = RSA.gen_key(key_size, exponent)

    privkey_filename = get_secure_filename()
    pubkey_filename = get_secure_filename()

    keyring.save_key(privkey_filename, callback=passphrase_callback)
    keyring.save_pub_key(pubkey_filename)

    privkey = RSA.load_key(privkey_filename, callback=passphrase_callback)
    pubkey = RSA.load_pub_key(pubkey_filename)

    os.unlink(privkey_filename)
    os.unlink(pubkey_filename)

    return (pubkey, privkey)

def create_certificate_authority(passphrase_callback, serial=0,
        notbefore=now(), notafter=plus_twentyyears(now()), digest="md5", **name):
    """Function to create a certificate authority.

    @param passphrase_callback: a callback that will be called without
    argument and that must return the passphrase you want to use
    on your CA key
    @type passphrase_callback: callable

    @param serial: Serial number for the certificate, default is 0
    @type serial: Integer

    @param notbefore: datetime.datetime instance when
    the certificate starts being valid, default is now
    @type notbefore: datetime.datetime instance

    @param notafter: datetime.datetime instance when the certificate
    stops being valid, default is 20 years more than now
    @type notafter: datetime.datetime instance

    @param: **name - The name of the subject of the request, possible
    arguments are:
    C     - Country name
    ST    - State or province name
    L     - Locality name
    O     - Organization name
    OU    - Organizational unit name
    CN    - Common name
    emailAddress - E-mail address

    """
    assert isinstance(notbefore, datetime.datetime)
    assert isinstance(notafter, datetime.datetime)

    pubkey, privkey = create_rsa_keyring(
            passphrase_callback=passphrase_callback)

    ca_priv_evp = as_evp_pkey(privkey)
    ca_pub_evp = as_evp_pkey(pubkey)
    ca_cert_req = create_cert_request(ca_pub_evp, ca_priv_evp, digest, **name)

    ca_cert = create_certificate(
            ca_cert_req,
            (ca_cert_req, ca_priv_evp),
            serial,
            (notbefore, notafter)
            )

    return ca_cert, pubkey, privkey

def create_x509_certificate(ca_cert, ca_privkey,
        cert_passphrase_callback, ca_passphrase_callback, serial=0, notbefore=now(),
        notafter=plus_twentyyears(now()), **name):
    """Create and return a new x509 certificate using a Certificate Authority

    @param ca_cert: The Certificate Authority to use to create
    the new X509 certificate
    @type ca_cert: M2Crypto.X509.X509 instance

    @param ca_privkey: The Certificate Authority private key, its the
    private key which was used for sign the CA
    @type ca_privkey: M2Crypto.RSA.RSA instance

    @param cert_passphrase_callback: A Python callable object that is invoked
    during key generation; its usual purpose is to provide visual
    feedback.
    @type cert_passphrase_callback: Callback

    @param ca_passphrase_callback:  A Python callable object that is invoked
    during the new certificate signature; its usual purpose is to provide visual
    feedback.

    @param serial: Serial number for the certificate, default is 0
    @type serial: Integer

    @param notbefore: datetime.datetime instance when the certificate
    starts being valid, default is 0
    @type notbefore: datetime.datetime instance

    @param notafter: datetime.datetime instance when the certificate
    stops being valid, default is 20 years
    @type notafter: datetime.datetime instance

    @param: **name - The name of the subject of the request, possible
    arguments are:
    C     - Country name
    ST    - State or province name
    L     - Locality name
    O     - Organization name
    OU    - Organizational unit name
    CN    - Common name
    emailAddress - E-mail address
    """
    
    pubkey, privkey = create_rsa_keyring(
        passphrase_callback=cert_passphrase_callback)

    priv_evp = as_evp_pkey(privkey)
    pub_evp = as_evp_pkey(pubkey)
    ca_privkey_evp = as_evp_pkey(ca_privkey)
    cert_req = create_cert_request(pub_evp, priv_evp, **name)
    cert = create_certificate(
            cert_req,
            (ca_cert, ca_privkey_evp),
            serial,
            (notbefore, notafter)
            )

    return cert, pubkey, privkey

def verify_x509_certificate(ca_cert, client_cert):
    """Verify if a certificate was signed by a certain CA.
    Return 1, if the CA certificate is ok otherwise 0

    @param ca_cert: The CA certificate to use to verify the client certificate
    @type ca_cert: M2Crypto.X509.X509 instance

    @param client_cert: The client certificate to verify
    @type client_cert: M2Crypto.X509.X509 instance

    @result: Return an integer 1 or 0
    """
    success = client_cert.verify(ca_cert.get_pubkey())

    return success

def encrypt(pubkey, clear_data):
    """Encrypt <clear_data> string with the public key from
    the certificate
    You must load a randpool file before calling this method with
    Rand.load_file('randpool', -1).
    After calling the method, you must save the randpool file
    with Rand.save_file('randpool')
    
    @param pubkey: The public key to use to encrypt the data,
    if you have an X509 certificate,
    you use the public key from certificate.get_pubkey() method
    to retrieve public key
    @type pubkey: M2Crypto.EVP.PKey instance

    @param clear_data: Clear data to encrypt
    @type clear_data: String

    @return: the encrypted data encoded with base64
    """
    rsapub = RSA.RSA_pub(pubkey.get_rsa().rsa)
    encrypted_data = rsapub.public_encrypt(clear_data, PADDING_TYPE)
    return base64.b64encode(encrypted_data)

def decrypt(privkey, encrypted_data):
    """Decrypt <encrypt_data> with the private key
    Encrypted data must be encode with base64
    
    @param privkey: The private RSA key to use for decryption
    @type privkey: M2Crypto.X509.X509 instance

    @param encrypt_data: Encrypted data (String) to decrypt,
    the encrypted data must be encoded with base64
    @type encrypt_data: String

    @return: the decrypted (clear) data
    """
    k = base64.b64decode(encrypted_data)
    clear_data = privkey.private_decrypt(k, PADDING_TYPE)
    return clear_data

def sign(privkey, data):
    """Sign the data and return the generated signature encoding with base64
    You must load a randpool file before calling this method with
    Rand.load_file('randpool', -1).
    After calling the method, you must save the randpool file with
    Rand.save_file('randpool')
    
    @param privkey: The private RSA key to use to sign the data
    @type privkey: M2Crypto.RSA.RSA instance

    @param data: The data string to sign
    @type data: String
    """
    priv_evp = as_evp_pkey(privkey)
    priv_evp.sign_init()
    priv_evp.update(data)
    sig = priv_evp.final()
    result = base64.b64encode(sig)
    return result
    
def sign_file(privkey, file_path):
    """Sign the data and return the generated signature encoding with base64
    
    @param privkey: The private RSA key to use to sign the data
    @type privkey: M2Crypto.RSA.RSA instance

    @param file_path: The path of file that contain data to sign
    @type file_path: String
    """
    priv_evp = as_evp_pkey(privkey)
    priv_evp.sign_init()
    f = open(file_path, 'rb')
    while True:
        buf = f.read(4096)
        if buf == '':
            break
        priv_evp.update(buf)

    f.close()
    sig = priv_evp.final()
    result = base64.b64encode(sig)

    return result

def verify(pubkey, data, signature):
    """Verify the data with the signature.
    signature must be encoded with base64

    @param pubkey: The public key to use to verify the signature with the data,
    if you have an X509 certificate, you use the public key from
    certificate.get_pubkey() method to retrieve public key
    @type pubkey: M2Crypto.EVP.PKey instance

    @param data: The data to verify with the signature
    @type file_path: String

    @param signature: The signature, the signature must be encoded in base64
    @type signature: String

    @result: Return 1 otherwise the method return an exception
    """
    M_sig = base64.b64decode(signature)
    pubkey.verify_init()
    pubkey.verify_update(data)
    result = pubkey.verify_final(M_sig)
    return result

def verify_file(pubkey, file_path, signature):
    """Verify the data with the signature.
    signature must be encoded with base64

    @param pubkey: The public key to use to verify the signature with the data,
    if you have an X509 certificate, you use the public key from
    certificate.get_pubkey() method to retrieve public key
    @type pubkey: M2Crypto.EVP.PKey instance

    @param file_path: The path of the file that contain data to verify
    with the signature
    @type file_path: String

    @param signature: The signature, the signature must be encoded in base64
    @type signature: String

    @result: Return 1 otherwise the method return an exception
    """
    M_sig = base64.b64decode(signature)

    pubkey.verify_init()
    f = open(file_path, 'rb')
    while True:
        buf = f.read(4096)
        if buf == '':
            break
        pubkey.verify_update(buf)

    f.close()
    result = pubkey.verify_final(M_sig)

    return result

